"""
GENERAL DESCRIPTION

PURPOSE
=======
Truebar.py was written as a proxy module between TB-listener and TB backend (TB-BE). The main purpose is to keep TB-listener
losely coupled and less dependant on direct specifics of the TB-BE. 

CLASSES
=======
All methods and attributes that are required for TB-listener to communicate with TB-BE are implemented within the class TruebarSTT. 
Additionaly, there are three other classes serving other purposes:

- TBError: a utility class to instantiate and log an error object in case an exception happens during invocation of any TruebarSTT method. 
- TranscriptReader: implements methods to support TB-explorer application
- TranscriptAligner: implements methods to support TB-explorer application

ERROR HANDLING
==============
Rules regarding exceptions: I only catch exceptions that I can handle meaningfully at this level. Otherwise I leave to the main program 
to catch them and do approriate actions. I use logging to record details about exceptions, even if I handle them. This can aid in debugging.

Author: VITASIS inc.
Date: May 2021
Last change: Feb 2024


TODO:
[ ] Except for the streaming and few other methods, the TruebarSTT class could be almost entairly implemented with one generic
    method. The parameters of the method should be:
    * request_data: dict: possible keys: url (mandatory), files, headers, data
    * needs_authentication: bool: if True, authentication is verified
    * log_exception: bool: if True, excepions are logged
    * return_err_object: bool: it True, an instance of TBError class is returned in case of any errors
    * return_type: string: could be json, content or object. It will return resp.json(), resp.content or resp respectively.
[ ] Check and remove transcribe_from_file and transcribe_from_mic if redundant
[x] Implement logging
[x] The library should never exit from application but rather get back to the caller whatever happened.
    It should return a message or an exception in case an error occured. Generic response should be a 
    dict with the following keys:
    - status: str: {'failed'|'success'}
    - status_code: int # if web service, status code as returned from the service otherwise None
    - message: str # whatever message you want to send back to the client.
    Appart from that, all important situations must be logged using logger.
[x] After long time of inactivity, authentication becomes invalid! This is not handled!!!
[ ] Imports should be distributed among classes.
[ ] Pynput library requires X server and DISPLAY to work appropriately. Will not run on headless servers. Consider replacing with keyboard - check pip install ketyborad - check below how the
    keyboard lib should be used

    import keyboard

    class YourClass:
        def __init__(self):
            keyboard.on_release(self.__on_keyrelease)

        def __on_keyrelease(self, key):
            # Private method to listen on keyboard for ESC or multimedia key PLAY/PAUSE to stop transcribing
            
            if key == keyboard.KeyCode.from_vk(179):  # VK_MEDIA_PLAY_PAUSE
                print('Multimedia key PLAY/PAUSE detected, stopping listener...')
                # stop listener
                self.__end_session()
                return False
            elif key == keyboard.Key.esc:
                print('\nESC key detected, stopping listener...')
                # stop listener
                self.__end_session()
                return False

        def __end_session(self):
            # Your code to end the transcription session
            pass

    YourClass()

    # Keep the program running
    keyboard.wait()


"""

import websocket
import threading
import time
import requests
import json
from pynput.keyboard import Key, Listener
from pynput import keyboard
import queue
import pyaudio
from requests.structures import CaseInsensitiveDict
from requests.exceptions import HTTPError, ConnectionError
from urllib3.exceptions import NewConnectionError
import codecs
import struct
import logging
from logging import config
import wave
import yaml
import os
import shutil
import re
from html2text import html2text
import socket
import certifi
import ssl
import uuid
import inspect
import copy

# ws close codes - defined by TB API
WEBSOCKET_CLOSE_CODES = [
    {'code': 1000, 'status': 'NORMAL', 'mtype': 'INFO', 'description': 'Normalno zapiranje seje'},
    {'code': 1001, 'status': 'GOING_AWAY', 'mtype': 'SERVER-ERROR', 'description': 'Zaledni sistem je v ponovnem zagonu'},
    {'code': 1002, 'status': 'PROTOCOL_ERROR', 'mtype': 'SERVER-ERROR', 'description': 'Napaka v komunikacijskem protokolu'},
    {'code': 1003, 'status': 'UNPROCESSABLE_DATA', 'mtype': 'SERVER-ERROR', 'description': 'Strežnik ne more obdelati podatkov, ker so v napačnem formatu'},
    {'code': 1007, 'status': 'UNPROCESSABLE_MSG', 'mtype': 'SERVER-ERROR', 'description': 'Strežnik ne more obdelati podatkov, ki so bila posredovana v sporočilu'},
    {'code': 1009, 'status': 'MSG_TOO_BIG', 'mtype': 'SERVER-ERROR', 'description': 'Strežnik ne more obdelati sporočila, ker je preveliko'},
    {'code': 1010, 'status': 'HANDSHAKE_PROBLEM', 'mtype': 'SERVER-ERROR', 'description': 'Povezava zavrnjena zaradi napačne vzpostavitve'},
    {'code': 1011, 'status': 'SERVER_ERROR', 'mtype': 'SERVER-ERROR', 'description': 'Neznana napaka na zalednem sistemu'},
    {'code': 4000, 'status': 'CLIENT_ERROR', 'mtype': 'SERVER-ERROR', 'description': 'Seja prekinjena zaradi napake na strani odjemalca'},
    {'code': 4001, 'status': 'CLIENT_ERROR_SILENCE', 'mtype': 'SERVER-ERROR', 'description': 'Seja prekinjena zaradi predolge tišine'},
    {'code': 4002, 'status': 'MODEL_IS_UPDATING', 'mtype': 'SERVER-ERROR', 'description': 'Poskus vzostavitve seje z modelom, ki je v prenovi'},
    {'code': 4030, 'status': 'ACCESS_DENIED', 'mtype': 'SERVER-ERROR', 'description': 'Uporabnik nima pravice za izvedbo zahtevane operacije'},
    {'code': 4031, 'status': 'MODEL_NOT_ALLOWED', 'mtype': 'SERVER-ERROR', 'description': 'Uporabnik nima pravice do izbranega modela'},
    {'code': 4041, 'status': 'MODEL_NOT_AVAILABLE', 'mtype': 'SERVER-ERROR', 'description': 'Izbran model ni na voljo'},
    {'code': 4291, 'status': 'WORKERS_NOT_AVAILABLE', 'mtype': 'SERVER-ERROR', 'description': 'Trenutno ni prostih procesov za izvedbo prepoznave govora'},
    {'code': 4292, 'status': 'QUOTA_EXCEEDED', 'mtype': 'SERVER-ERROR', 'description': 'Uporabnik je presegel dovoljeno kvoto generiranja transkriptov'},
    {'code': 4293, 'status': 'CONCURRENCY_LIMIT_EXCEEDED', 'mtype': 'SERVER-ERROR', 'description': 'Uporabnik je presegel dovoljeno število sočasnih sej'}, 
]

def setup_logging(logger_name, logger_folder):
    """
    Read logger config and make sure to use correct logger folder
    """
    with open('./conf/config-logger.yaml', 'r', encoding="utf-8") as stream:
        cfg = yaml.load(stream, Loader=yaml.FullLoader)
    cfg['handlers']['info_file_handler']['filename'] = os.path.join(logger_folder, 'info.log')
    cfg['handlers']['debug_file_handler']['filename'] = os.path.join(logger_folder, 'debug.log')
    cfg['handlers']['error_file_handler']['filename'] = os.path.join(logger_folder, 'error.log')
    logging.config.dictConfig(cfg)
    return logging.getLogger(logger_name)

class TBError():
    
    def __init__(
            self,
            error_code,
            error_message,
            exception,
            error_type,
            error_ts=time.time(),
            error_id=str(uuid.uuid4()),
        ):
        # Get information about the calling function
        caller_frame = inspect.stack()[1]
        caller_function_name = caller_frame[3]

        self.caller = caller_function_name
        self.error_code = error_code
        self.error_message = error_message
        self.exception = exception
        self.error_type = error_type
        self.error_ts = error_ts
        self.error_id = error_id

class TranscriptReader():
    def __init__(
        self, 
        allowed_commands=[],
        allowed_punctuations=[],
        ):

        # dictionary that holds current state of tags
        self.tag_state=dict()
        # a list of token elements, each element has a text, spaceBefore tag, start and end time ...
        self.transcript_tokens=[]
        # a list of clean tokes
        self.clean_tokens=[]

        # command to html
        self.command_html_translation = {
            '<b>': '<b>',
            '</b>': '</b>', 
            '<i>': '<i>',
            '</i>': '</i>',
            '<u>': '<u>',
            '</u>': '</u>',
            '<uc>': '<div style="text-transform: uppercase;">',
            '</uc>': '</div>',
            '<lc>': '<div style="text-transform: lowercase;">',
            '</lc>': '</div>',
            '<alignl>': '<p align="left">',
            '<alignr>': '<p align="right">',
            '<alignc>': '<p align="center">',
            '<nl>': '<br>',
            '<np>': '<br><br>',
        }

        # commands that are not allowed will be ignored
        if len(allowed_commands)>0:
            self.allowed_commands = allowed_commands
        else:
            self.allowed_commands=['<nl>', '<np>', '<b>', '</b>', '<i>', '</i>', '<u>', '</u>', '<alignl>', '<alignr>', '<alignc>', '<uc>', '</uc>', '<lc>', '</lc>', '<spc>']

        # punctuation commands that are not allowed will be ignored
        if len(allowed_punctuations)>0:
            self.allowed_punctuations = allowed_punctuations
        else:
            # TODO: check with the puct in database
            self.allowed_punctuations=['<.>', '<,>', '<!>', '<?>', '<:>', '<;>', '<->', '<–>', '<%>', '<°>', '<+>', '<">', '<×>']

        # regex pattern for extracting punct or command
        self._cmd_ptr = re.compile(r"^(<)(.*)(>)$")

    def _command_to_html(self, command):
        """
        Returns html code that implements action of the command. 
        If no translation is found, empty string is returned
        """ 
        return self.command_html_translation.get(command['text'], "")
        
    def _punctuation_to_html(self, punctuation):
        """
        Returns html code that corresponds to the punctuation command
        """
        punct = re.sub(self._cmd_ptr, r"\g<2>", punctuation['text'])
        
        if punctuation.get('spaceBefore', False):
            return " "+punct
        else:
            return punct

    def html_to_text(self, html):
        """
        Exrats text from the html
        """
        return html2text(html)

    def _parse_transcript(self, transcript_with_metadata):
        """
        Takes transcript in the form as returned from the ASR Kaldi and
        creates a data structure 'transcript_tokens' of the TranscriptReader class  
        TODO: 
        [ ] remove not needed metadata
        """
        self.transcript_tokens=[]
        self.clean_tokens=[]
        cdx = 0
        for ide in transcript_with_metadata:
            content = json.loads(ide['content'])
            for el in content:
                self.transcript_tokens.append(el)
                self.clean_tokens.append(el["text"])

    def transcript_to_html(self, transcript):
        """
        Takes transcript in the form as returned from the ASR Kaldi and 
        returns corresponding html code.
        TODO: TB-BE returned structure includes key 'metadata'
        """
        self._parse_transcript(transcript) 
        text = ""
        for cdx, t in enumerate(self.transcript_tokens):
            if t['text'] in self.allowed_punctuations:
                text += self._punctuation_to_html(t)
            elif t['text'] in self.allowed_commands:
                text += self._command_to_html(t)
            elif re.match(self._cmd_ptr, t['text']):
                pass
            else:
                if t['spaceBefore'] or \
                    (cdx > 0 and f"<{self.transcript_tokens[cdx-1]['text']}>" in self.allowed_punctuations and \
                        (not t['text'].isnumeric() or not self.transcript_tokens[cdx-1]['text'].isnumeric())
                    ):
                    text += " " + t['text']
                else:
                    text += t['text']
            
        return text

class TranscriptAligner():
    def __init__(
        self, 
        content, 
        transcript, 
        logger_folder=os.path.dirname(__file__)
        ):
        self.content = content
        self.transcript = transcript
        self.visible_tokens = []
        self.visible_tokens_original = []
        # update transcript with absolute positions of tokens in the text (content)
        self._update_transcript_with_pos()
        # multiple space pattern
        self._MULTIPLE_SPACES_PTRN = re.compile(r'\s{2,}')
        # set logger
        self.logger = setup_logging("TB_aligner", logger_folder)

    def _update_transcript_with_pos(self):
        """
        Updates the transcript list with absolute positions of tokens in the content.
        At the same time it creates visible_tokens list that contains only tokens that are 
        visible in html (ignoring those with pos=-1, i.e. commands). This structure is 
        lighter than the transcript data structure which includes many other metadata and 
        all tokens. The visible_tokens structre is dynamic (it reflects all changes made on 
        the original transcript) and is used in find_tokens_by_pos to locate tokens when 
        synching with audio. Absolute token positions are altered if editor content is
        changed. Check function update_tokens_pos()

        To facilitate tracking changes, a copy of visible tokens is created which is immutable
        and will always reflect the original transcript. 
        
        Args:
        - None
        """
        current_position = 0
        nl_before = False
        np_before = False

        for c, el in enumerate(self.transcript):
            nl_before = True if c > 0 and self.transcript[c-1]['text'] == '<nl>' else False
            np_before = True if c > 0 and self.transcript[c-1]['text'] == '<np>' else False

            position_in_text = self.content.find(el['text'], current_position)
            el.update(pos=position_in_text)
            if position_in_text != -1:
                self.visible_tokens.append(dict(abs_pos=c, text_orig=el["text"], text=el["text"], startTime=el["startTime"], endTime=el["endTime"], pos=position_in_text, nl_before=nl_before, np_before=np_before))
                current_position = position_in_text + len(el['text'])
        
        self.visible_tokens_original = copy.deepcopy(self.visible_tokens)
        #print("INITIAL")
        #print(self.visible_tokens)

    def update_tokens_pos(self, pos_of_change, change, new_text=None):
        """
        Call this function to change pos of transcript tokens if editor content changes
        All transcript tokens that have start_pos >= pos_of_change, must be altered to 
        take into account the change
        In addition, alter text key of the changed token(s) to reflect the change
        """

        #print(pos_of_change, change, new_text)
        
        # find token, where the change was made
        changed_element_pos = self.find_token_by_pos(pos_of_change, 0, len(self.visible_tokens))

        if not changed_element_pos[4]:
            self.logger.warning("Can't find token where change was made!")
            # The most probable reasons for not being able to find the token of change is that the cursor is in the 
            # middle of two spaces. This happens when a complete word is deleted and spaces of two consecutive words are joined.
            # The cursor in the middle of two spaces truly doesn't belong to any token. In this situation do the following:
            # a) if ADD was pressed, add new token in that position and move all next tokens for the lenght of the added token
            # b) if REM was pressed, the token will not be found only in the case one character was deleted - the one that doesn't 
            # belong anywhere. So it is safe to just reposition all next tokens for one character left (notice that change will be -1).
            if change > 0:
                
                # if inserting on position 0, use specific abs_pos and insert pos
                abs_pos_new = 1 if pos_of_change == 0 else self.visible_tokens[changed_element_pos[3]]["abs_pos"] + 1
                insert_pos = 0 if abs_pos_new == 1 else changed_element_pos[3]+1
                
                # set start and end time to the end_time of the previous element or to the start time of the currently first element if insert_pos=0
                if insert_pos == 0:
                    element_time = self.visible_tokens[0]["startTime"]
                else:
                    element_time = self.visible_tokens[insert_pos-1]["endTime"]
                
                self.visible_tokens.insert(insert_pos, dict(abs_pos=abs_pos_new, text=new_text, text_orig="", startTime=element_time, endTime=element_time, pos=pos_of_change))
                
                for el in self.visible_tokens[changed_element_pos[3]+2:]:
                    el["pos"] += len(new_text)
                #print([f"{s['text']} - {s['pos']}" for s in self.visible_tokens[changed_element_pos[3]-5:changed_element_pos[3]+5]])
            else:
                for el in self.visible_tokens[changed_element_pos[3]+1:]:
                    el["pos"] += change
                #print([f"{s['text']} - {s['pos']}" for s in self.visible_tokens[changed_element_pos[3]-5:changed_element_pos[3]+5]])
            return
        else:
            changed_element_pos = changed_element_pos[3]
        
        #print("BEFORE:")
        #print([f"{s['text']} - {s['pos']}/{s['abs_pos']}" for s in self.visible_tokens[changed_element_pos-5:changed_element_pos+5]])
        #print([f"{s}\n" for s in self.visible_tokens[changed_element_pos-5:changed_element_pos+5]])

        change_span = abs(change)
        previous_pos = 0

        for pos_idx, el in enumerate(self.visible_tokens[changed_element_pos:]):
            current_pos = el['pos']
            current_el_len = len(el["text"])
            
            # CHARACTERS REMOVED
            if change < 0:
                # if first element - the element where the change starts - change it accordingly. Pos of this element should not change except if it was deleted completely
                if pos_idx == 0:
                    start_el_pos = current_pos
                    el["text"] = el["text"][:pos_of_change-current_pos] + el["text"][pos_of_change-current_pos+abs(change):]
                    # set pos to -1 if token was deleted completely
                    if abs(change) >= current_el_len and pos_of_change == current_pos:
                        el["pos"] = -1
                else:
                    # decrease change span if larger than the previous element, or set to 0
                    change_span = max(0, change_span - (current_pos - previous_pos)) 
                    # if change_span > 0 (this happens when text is selected and deleted at once) - apply further changes
                    if change_span > 0:
                        if change_span >= current_el_len:
                            el["pos"] = -1
                            el["text"] = ""
                        else:
                            # this is last element of the change spanning over multiple tokens
                            el["pos"] = pos_of_change
                            el["text"] = el["text"][change_span:]
                    # if the element out of the change span, change its pos for the "change"
                    else:
                        el["pos"] = el["pos"] + change

            # CHARACTERS ADDED
            else:
                #print(pos_of_change, change, new_text)
                # all new text is added to the element where change starts. This element remains the same starting pos
                if pos_idx == 0:
                    el["text"] = el["text"][:pos_of_change-current_pos] + new_text + el["text"][pos_of_change-current_pos:]
                # all other elements remain the same except that they change pos
                else:
                    el["pos"] = el["pos"] + change

            # if this is the first element, set previous pos to position of change, otherwise set it to the position of current element
            previous_pos = current_pos if pos_idx > 0 else pos_of_change

            
        # purge list of visible tokens by removing tokens that have been deleted
        self.visible_tokens = [el for el in self.visible_tokens if el['pos']!=-1]

        #print("AFTER")
        #print([f"{s['text']} - {s['pos']}/{s['abs_pos']}" for s in self.visible_tokens[changed_element_pos-5:changed_element_pos+5]])
        #print([f"{s}\n" for s in self.visible_tokens[changed_element_pos-5:changed_element_pos+5]])

    def _remove_punctuations(
        self,
        text,
        punctuations = ".,!?:;" 
        ):
        """
        This will remove all punctuations, even dot symbols after abbreviations.
        To deal with abbreviations differently, use 'remove_punctuations' method 
        from the Vitasis.Extractor class!
        """
        return re.sub(rf"[{punctuations}]", "", text)

    def _remove_multiple_spaces(self, t):
        """
        Converts multiple into a single space
        """
        return re.sub(self._MULTIPLE_SPACES_PTRN, " ", t)

    def align_text(
        self, 
        text1, 
        text2, 
        convert_line_breaks=True,
        remove_punctuation=True, 
        lowercase=True,
        ):
        """
        Aligns tokens from text1 and text2. Returns results where both texts are 
        represented with a list of their tokens except in places where the alignment
        was not posible. In such places "-" char is used to indicate that at that
        position alignment was not established.
        Args:
        - text1: first text
        - text2: second text
        - convert_line_breaks: if True, line breaks \n will be converted to a single space
        - remove_punctuation: if True, punctuations are removed
        - lowercase: if True, texts are first lowercased
        """
        match_award      = 20
        mismatch_penalty = -5
        gap_penalty      = -5 # both for opening and extanding

        def zeros(shape):
            retval = []
            for x in range(shape[0]):
                retval.append([])
                for y in range(shape[1]):
                    retval[-1].append(0)
            return retval

        def match_score(alpha, beta):
            if alpha == beta:
                return match_award
            elif alpha == '-' or beta == '-':
                return gap_penalty
            else:
                return mismatch_penalty

        def finalize(align1, align2):
            align1.reverse()    #reverse sequence 1
            align2.reverse()    #reverse sequence 2
            
            i,j = 0,0
            
            #calcuate identity, score and aligned sequeces

            found = 0
            score = 0
            identity = 0
            for i in range(0,len(align1)):
                # if two AAs are the same, then output the letter
                if align1[i] == align2[i]:                

                    identity = identity + 1
                    score += match_score(align1[i], align2[i])
            
                # if they are not identical and none of them is gap
                elif align1[i] != align2[i] and align1[i] != '-' and align2[i] != '-': 
                    score += match_score(align1[i], align2[i])
                    found = 0
            
                #if one of them is a gap, output a space
                elif align1[i] == '-' or align2[i] == '-':          
                    score += gap_penalty
            
            identity = float(identity) / len(align1)
            
            return identity, score, align1, align2

        def needle(seq1, seq2):
            m, n = len(seq1), len(seq2)  # length of two sequences
            
            # Generate DP table and traceback path pointer matrix
            score = zeros((m+1, n+1))      # the DP table
        
            # Calculate DP table
            for i in range(0, m + 1):
                score[i][0] = gap_penalty * i
            for j in range(0, n + 1):
                score[0][j] = gap_penalty * j
            for i in range(1, m + 1):
                for j in range(1, n + 1):
                    match = score[i - 1][j - 1] + match_score(seq1[i-1], seq2[j-1])
                    delete = score[i - 1][j] + gap_penalty
                    insert = score[i][j - 1] + gap_penalty
                    score[i][j] = max(match, delete, insert)

            # Traceback and compute the alignment 
            align1, align2 = [], []
            i,j = m,n # start from the bottom right cell
            while i > 0 and j > 0: # end toching the top or the left edge
                score_current = score[i][j]
                score_diagonal = score[i-1][j-1]
                score_up = score[i][j-1]
                score_left = score[i-1][j]

                if score_current == score_diagonal + match_score(seq1[i-1], seq2[j-1]):
                    align1.append(seq1[i-1])
                    align2.append(seq2[j-1])
                    i -= 1
                    j -= 1
                elif score_current == score_left + gap_penalty:
                    align1.append(seq1[i-1])
                    align2.append('-')
                    i -= 1
                elif score_current == score_up + gap_penalty:
                    align1.append('-')
                    align2.append(seq2[j-1])
                    j -= 1

            # Finish tracing up to the top left cell
            while i > 0:
                align1.append(seq1[i-1])
                align2.append('-')
                i -= 1
            while j > 0:
                align1.append('-')
                align2.append(seq2[j-1])
                j -= 1

            return finalize(align1, align2)

        def water(seq1, seq2):
            m, n = len(seq1), len(seq2)  # length of two sequences
            
            # Generate DP table and traceback path pointer matrix
            score = zeros((m+1, n+1))      # the DP table
            pointer = zeros((m+1, n+1))    # to store the traceback path
            
            max_score = 0        # initial maximum score in DP table
            # Calculate DP table and mark pointers
            for i in range(1, m + 1):
                for j in range(1, n + 1):
                    score_diagonal = score[i-1][j-1] + match_score(seq1[i-1], seq2[j-1])
                    score_up = score[i][j-1] + gap_penalty
                    score_left = score[i-1][j] + gap_penalty
                    score[i][j] = max(0,score_left, score_up, score_diagonal)
                    if score[i][j] == 0:
                        pointer[i][j] = 0 # 0 means end of the path
                    if score[i][j] == score_left:
                        pointer[i][j] = 1 # 1 means trace up
                    if score[i][j] == score_up:
                        pointer[i][j] = 2 # 2 means trace left
                    if score[i][j] == score_diagonal:
                        pointer[i][j] = 3 # 3 means trace diagonal
                    if score[i][j] >= max_score:
                        max_i = i
                        max_j = j
                        max_score = score[i][j];
            
            align1, align2 = [], []    # initial sequences
            
            i,j = max_i,max_j    # indices of path starting point
            
            #traceback, follow pointers
            while pointer[i][j] != 0:
                if pointer[i][j] == 3:
                    align1.append(seq1[i-1])
                    align2.append(seq2[j-1])
                    i -= 1
                    j -= 1
                elif pointer[i][j] == 2:
                    align1.append('-')
                    align2.append(seq2[j-1])
                    j -= 1
                elif pointer[i][j] == 1:
                    align1.append(seq1[i-1])
                    align2.append('-')
                    i -= 1

            return finalize(align1, align2)
    
        if remove_punctuation:
            text1 = self._remove_punctuations(text1)
            text2 = self._remove_punctuations(text2)

        if lowercase:
            text1 = text1.lower()
            text2 = text2.lower()

        if convert_line_breaks:
            text1 = text1.replace("\n", " ")
            text2 = text2.replace("\n", " ")

        aligned_data = needle(
            self._remove_multiple_spaces(text1).split(" "), 
            self._remove_multiple_spaces(text2).split(" ")
            )

        return aligned_data

    def create_op_data(self, asr_text, ref_text, with_timings=True):
        """
        Compares ASR text with some referential text and returns data structure 
        with information on the alignment. The structure includes for every element 
        in the aligned list:
        - operation: 
        - token from text1
        - token from text2
        - token absolute start time
        - token duration
        - pause before token
        - pause after token
        The last four parameters are added if 'with_timings' is True
        Args:
        - asr_data: ASR text
        - ref_data: REF text
        - with_timings: if True, time attributes (start time, duration, pause before, pause after) 
          are added to the resulting data structure
        Results:
        - a list with op data
        - a list with errors 
        """

        def get_value(elm_id, phase):
            return float(self.transcript[elm_id][phase])

        def count_INS(transcript):
            return len([el for el in transcript if el=='-'])

        #asr_text = "Danes je lep sončen dan."
        #ref_text = "Dane jep lep sončen dan"

        res = self.align_text(asr_text.strip(), ref_text.strip())

        offset = 0

        op_data = []
        error_data = []
        deleted_tokens = 0

        for i in range(0, len(res[2])):
            if res[2][i] == res[3][i]:
                op = 'OK'
            elif res[2][i]=='-':
                op='INS'
                offset -= 1
            elif res[3][i]=='-':
                op='DEL'
            else:
                op='SUB'

            if res[2][i] not in ('-', self._remove_punctuations(self.transcript[i+offset]['text'].lower())):
                error_data.append(dict(
                    error='no matching token in the asr timings',
                    ref_token=res[2][i],
                    asr_token=res[3][i],
                    asr_timings_token=self.transcript[i+offset]['text']
                    ))
                
            if with_timings:
                try:
                    w_pos = get_value(i+offset, 'pos')
                    w_start = get_value(i+offset, 'startTime')
                    w_end =  get_value(i+offset, 'endTime')
                    w_len = round(w_end - w_start, 3)
                    
                    if op=='DEL':
                        deleted_tokens+=1
                    
                    if i==0:
                        w_pause_before = 0
                    else:
                        w_pause_before = round(w_start - get_value(i+offset-1, 'endTime'), 3)
                    
                    if i+offset>=len(self.transcript)-1:
                        w_pause_after = 0
                    else:
                        w_pause_after = round(get_value(i+offset+1, 'startTime') - w_end, 3)
                        
                    op_data.append(dict(
                        operation=op,
                        ref_token=res[3][i],
                        asr_token=res[2][i],
                        start_time=w_start,
                        duration=w_len,
                        pause_before=w_pause_before,
                        pause_after=w_pause_after,
                        pos=w_pos,
                        pos_index=i-deleted_tokens
                    ))
                    
                except Exception as e:
                    self.logger.error(f"Error when creating OP file. Exception: {e}")
                    self.logger.error(f"asr_roken={res[2][i]}, ref_token={res[3][i]}, step={i}, offset={offset}, asr_timings_token={self.transcript[i+offset]['text']}")

            else:
                op_data.append(dict(
                        operation=op,
                        ref_token=res[3][i],
                        asr_token=res[2][i],
                    ))

        return op_data, error_data

    def find_spoken_token(self, ct, from_el, to_el):
        """
        Returns token that is spoken at time 'ct'. Bisection is used to make search fast.
        Args:
        - ct: time in miliseconds
        - from_el: id of the element to look from
        - to_el: id of the element to look to
        """
        
        def get_correct_el(ct, fe, me, te):
            # get rid of duplicates
            ellist = set([fe, me, te])
            for el in ellist:
                print(el, ct, len(self.visible_tokens))
                print("*************************", ct, self.visible_tokens[el]['text'], self.visible_tokens[el]['endTime'])
                if ct <= self.visible_tokens[el]['endTime']:
                    return el
            return None

        mid_el_num = from_el + int((to_el - from_el) / 2)

        if from_el == to_el or from_el==mid_el_num or to_el==mid_el_num:
            return None
            bestel = get_correct_el(ct, from_el, mid_el_num, to_el)
            if bestel:
                return (
                    self.visible_tokens[bestel]['text'], 
                    self.visible_tokens[bestel]['pos']
                    )
            else:
                return None

        mid_ts_start = self.visible_tokens[mid_el_num]['startTime']
        mid_ts_end = self.visible_tokens[mid_el_num]['endTime']

        if ct < mid_ts_start:
            return self.find_spoken_token(ct, from_el, mid_el_num)
        elif ct <= mid_ts_end:
            return (
                self.visible_tokens[mid_el_num]['text'], 
                self.visible_tokens[mid_el_num]['pos']
                )
        else:
            return self.find_spoken_token(ct, mid_el_num, to_el)

    def find_token_by_pos(self, pos, from_el, to_el):
        """
        Returns text, startTime and endTime of the token that is positioned at position 
        pos in the current editor content
        Args:
        - pos: absolute position in the content
        - from_el: id of the element to look from
        - to_el: id of the element to look to

        Returns a tuple with the following element:
        - text of the found token
        - start time of the found token
        - end time of the found token
        - mid-element number
        - True if token was found or False otherwise. If false, all other values of the tuple belong to the last mid_element
        """
        mid_el_num = from_el + int((to_el - from_el) / 2)
        if len(self.visible_tokens)-1 < mid_el_num:
            print("Got element ", mid_el_num, " not in the list!") 
        mid_el_start_pos = self.visible_tokens[mid_el_num]['pos']
        mid_el_end_pos = mid_el_start_pos + len(self.visible_tokens[mid_el_num]['text'])

        if from_el == to_el or from_el==mid_el_num or to_el==mid_el_num:
            # check if the last bisection element is the correct one - if not, the element for the pos cannot be found
            if pos >= mid_el_start_pos and pos <= mid_el_end_pos:
                return (
                    self.visible_tokens[mid_el_num]['text'], 
                    self.visible_tokens[mid_el_num]['startTime'],
                    self.visible_tokens[mid_el_num]['endTime'],
                    mid_el_num,
                    True
                )
            else:
                self.logger.warning(f"Can't find token for pos: {pos}. Assuming cursor is in the middle of two spaces ...")
                return (
                    self.visible_tokens[mid_el_num]['text'], 
                    self.visible_tokens[mid_el_num]['startTime'],
                    self.visible_tokens[mid_el_num]['endTime'],
                    mid_el_num,
                    False
                )

        if pos < mid_el_start_pos:
            return self.find_token_by_pos(pos, from_el, mid_el_num)
        elif pos <= mid_el_end_pos:
            return (
                self.visible_tokens[mid_el_num]['text'], 
                self.visible_tokens[mid_el_num]['startTime'],
                self.visible_tokens[mid_el_num]['endTime'],
                mid_el_num,
                True
                )
        else:
            return self.find_token_by_pos(pos, mid_el_num, to_el)

    def index_tokens(self, plaintext):
        """
        Returns a list with absolute positions of tokens in the plaintext.
        Args:
        - plaintext: text to index
        """
        #return [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        pos = [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        # remove positions for multiple spaces - only last space in the row is relevant
        pos_cleaned = []
        for i in range(0, len(pos)):
            if i<len(pos)-1 and pos[i]==pos[i+1]:
                continue
            else:
                pos_cleaned.append(pos[i])
        return pos_cleaned

    def token_positions(self, plaintext):
        """
        """
        indexed_tokens = dict()
        # get positions of spaces
        pos = [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        # remove positions for multiple spaces - only last space in the row is relevant
        pos_cleaned = []
        for i in range(0, len(pos)):
            if i<len(pos)-1 and pos[i]==pos[i+1]:
                continue
            else:
                pos_cleaned.append(pos[i])
        # create dict, keys must be without punctuations
        plaintext_cleaned = self._remove_multiple_spaces(plaintext)
        for c, token in enumerate(plaintext_cleaned.split(" ")):
            indexed_tokens[self._remove_punctuations(token)]=pos_cleaned[c]

        return indexed_tokens

class TruebarSTT():

    def __init__(
        self, 
        TB_config, 
        logger_folder=os.path.dirname(__file__)
        ):
        self.token_attributes = {}
        self.configuration_attributes = {}
        self.session = None
        self.last_sessionId = None
        self.session_start = None
        self.elapsed_time_to_connect = None # tells how much time was needed last time to connect - this is kind of ping
        self.session_info = []         # keeps info about all sessions performed with the object
        self.connection_errors = []    # keeps info about failed attempts if any during establishing the last session
        self.TB_config = TB_config
        self.sender = None
        self.listener = None
        self.state = None              # added to keep info on session state. Could be one of the following: {None, Authenticated, Connected, Error}
        self.group_users = []
        self.timeout = self.TB_config.pop('timeout', None)
        self.data = queue.Queue()
        self.status = queue.Queue()
        self.microphone_stream = None
        self.input_devices = self.get_input_devices()
        self.selected_input_device_index = None   # if None, default system input will be used
        self.logger = setup_logging("TB_library", logger_folder)
        self.truecaser_url = 'http://88.200.23.57:50011/v1/truecase'
        self.truecaser_tokens_url = 'http://88.200.23.57:50011/v1/truecase_tokens'
        self.http_protocol = 'https'    # by default secure protocol HTTPS is used. If local server is selected at authentication, the protocol will be changed to HTTP
        """
        class initialization
        """

        # register close codes
        self.ws_close_codes = dict()
        for el in WEBSOCKET_CLOSE_CODES:
            self.ws_close_codes[el['code']]={'status': el['status'], 'mtype': el['mtype']}

    def get_ip(self):
        """
        Return IP of the local machine
        """
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.settimeout(0)
        try:
            # doesn't even have to be reachable
            s.connect(('10.254.254.254', 1))
            IP = s.getsockname()[0]
        except Exception:
            IP = '127.0.0.1'
        finally:
            s.close()
        return IP
        
    def local_root_certificate(self):
        """
        check for local Root certificate if exists
        """
        if os.path.isfile(os.path.join(os.path.dirname(__file__), 'cert', 'ISRGRootX1.pem')):
            return os.path.join(os.path.dirname(__file__), 'cert', 'ISRGRootX1.pem')
        else:
            return True

    def get_auth_token(self):
        """
        authenticate to get a token        
        """
        url = self.TB_config.get("auth-url", None)
        
        if self.TB_config.get("use_local_server", False):
            verify = self.local_root_certificate()
            if verify != True:
                self.logger.debug(f"Verification of root certificate via local file at {verify}")
        else:
            verify = False
        
        headers = CaseInsensitiveDict()
        headers["Content-Type"] = "application/x-www-form-urlencoded"
        headers["Content-Length"] = "0"
        data = "grant_type=password&username="+self.TB_config["username"]+"&password="+self.TB_config["password"]+"&client_id=truebar-client"

        try:
            resp = requests.post(url, headers=headers, data=data, verify=verify)
        except Exception as err:
            err_obj = TBError(error_code=10061, error_message="Local host not available", error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            self.connection_errors.append(vars(err_obj))
            return False

        try:
            resp.raise_for_status()
            self.token_attributes = resp.json()
            self.state = 'Authenticated'
            return resp.json()["access_token"]

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            self.connection_errors.append(vars(err_obj))
            return False

    def user_authenticated(self):
        """
        Check if user is authenticated
        """
        if not self.token_attributes.get("access_token", False):
            return False
        else: 
            return True

    def get_input_devices(self):
        """
        Get info about available input devices
        """
        input_devices = dict()
        p = pyaudio.PyAudio()
        info = p.get_host_api_info_by_index(0)
        numdevices = info.get('deviceCount')
        for i in range(0, numdevices):
            if (p.get_device_info_by_host_api_device_index(0, i).get('maxInputChannels')) > 0:
                device_name = p.get_device_info_by_host_api_device_index(0, i).get('name')
                if re.search(r"^microsoft sound mapper", device_name.lower()):
                    continue
                input_devices[i] = p.get_device_info_by_host_api_device_index(0, i).get('name')
                #print("Input Device id ", i, " - ", p.get_device_info_by_host_api_device_index(0, i).get('name'))
        return input_devices

    def get_configuration(self):
        """
        Read configuration /must be authenticated
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
        
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + "/api/client/configuration"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            self.configuration_attributes = resp.json()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_available_models(self):
        """
        Check available model. Returns a list of tuples (framework, language, domain, model)
        """
        stt_status = self.get_stt_status()
        available_models = []

        if stt_status:
            for f in stt_status["frameworks"]:
                if not f["isAllowed"]:
                    continue
                fcode = f["code"]
                for l in f["languages"]:
                    if not l["isAllowed"]:
                        continue
                    lcode = l["code"]
                    for d in l["domains"]:
                        if not d["isAllowed"]:
                            continue
                        dcode = d["code"]
                        for m in d["models"]:
                            if not m["isAllowed"]:
                                continue
                            mcode = m["code"]
                            available_models.append(dict(
                                framework=fcode,
                                language=lcode,
                                domain=dcode,
                                model=mcode
                            ))
        else:
            self.logger.warning(f"The user doesn't have access to any framework!")
            return []
        return available_models

    def get_stt_status(self):
        """
        Get session status
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + "/api/stt/status"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        headers["Content-Type"] = "application/json"

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_model_versions(self, langcode, domaincode):
        """
        Public method to check model versions for a specific language and domain
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        config = self.get_configuration()
        versions = []
        for lang in config.get("transcriptionLanguageOptions"):
            if lang["languageCode"]==langcode:
                for domain in lang.get("domains"):
                    if domain["domainCode"]==domaincode:
                        for ver in domain["models"]:
                            if ver["allowed"] and ver["available"]:
                                versions.append(ver["version"])
        return versions
    
    def update_configuration(self, new_config):
        """
        Public method to update configuration
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + "/api/client/configuration"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        headers["Content-Type"] = "application/json"

        data = new_config

        resp = requests.patch(url, headers=headers, data=data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
    
    def set_model(self, framework, lang, domain, version):
        """
        Public method to to set language, domain and model version
        """
        new_config = dict(
            stt=dict(
                framework=dict(
                    value=framework
                ),
                language=dict(
                    value=lang
                ),
                domain=dict(
                    value=domain
                ),
                model=dict(
                    value=version
                )
            )
        )
        self.update_configuration(json.dumps(new_config))
        return True

    # start transcription from file or microphone ...
    def transcribe(self, input="microphpne", audiofile=None, outfile=None):
        """
        Generic public method to start transcription session
        """
        self.__start_session(input, audiofile, outfile)

    def transcribe_from_microphone(self, continue_session=False, keyListener=True, ignore_ssl=False):
        """
        Public method to start transcription session from microphone
        """
        return self.__start_session(keyListener=keyListener, continue_session=continue_session, ignore_ssl=ignore_ssl)

    def __microphone_streamer(self):
        """
        Private method to stream from microphone
        """
        # initiate microphone stream
        audio = pyaudio.PyAudio()
        self.stream = audio.open(format=self.TB_config["audio_format"],
                    channels=self.TB_config["audio_channels"],
                    rate=self.TB_config["audio_rate"],
                    input=True,
                    input_device_index=self.selected_input_device_index,
                    frames_per_buffer=self.TB_config["audio_chunk"])
        # enqueue chunks
        while self.session and self.session.connected:
        #while self.session.sock and self.session.sock.connected:
            data = self.stream.read(self.TB_config["audio_chunk"])
            self.data.put(data)

        # use explicit return, otherwise the thread will not close
        return True

    def __enqueue_wave_frames(self, audiofile):
        """
        Private method to enqueue wave file frames
        """
        # enqueue frames
        with wave.open(audiofile, 'rb') as wf:
            data = wf.readframes(self.TB_config["audio_chunk"])
            while data != b"": #and self.session and self.session.connected:
                self.data.put(data)
                data = wf.readframes(self.TB_config["audio_chunk"])
                if self.TB_config.get("simulate_realtime_speech", False): 
                    time.sleep((len(data) / self.TB_config["audio_rate"])/2)
            self.data.put(b"")
            self.logger.info("file enqueued")

        # use explicit return, otherwise the thread will not close
        return True
                   
    def __data_sender(self):
        """
        Private method to read data from a queue and send it to the ASR server. The queue is fed by a microphon stream or from a file.
        """
        while self.session and self.session.connected:
        #while self.session.sock and self.session.sock.connected:
            try:
                data = self.data.get(block=True, timeout=self.timeout)
                # if reading from file and b"" is encountered, end the session
                if data == b"":
                    self.__end_session()
            except queue.Empty:
                if not self.session or not self.session.connected:
                    response = dict(messageType="ERROR", message="session failure and no chunks in queue")
                    self.status.put(response)
            else:
                if not data or not self.session:
                    break
                try:
                    self.session.send_binary(bytes(data))
                except Exception as e:
                    response = dict(status=4, message=f"session failure, {e}")
                    response = dict(messageType="ERROR", message=f"session failure, {e}")
                    self.status.put(response)
        
        # use explicit return, otherwise the thread will not close
        return True
    
    def __response_listener(self):
        """
        Private method that listens to responses, received from the ASR
        """
        while self.session and self.session.connected:
            try:
                #msg = self.session.recv()
                resp_opcode, msg = self.session.recv_data()
                if resp_opcode == 8 and len(msg) >= 2:
                    """
                    close_code = str(struct.unpack("!H", msg[0:2])[0])
                    close_message = msg[2:].decode("utf-8")
                    self.session_info.append(dict(messageType='INFO', close_code=close_code, close_message=close_message))
                    self.logger.debug("websocket closed with code: %s, message: %s" %(close_code, close_message))
                    self.session = None
                    """
                    self.__process_closed_ws_response(msg)
                    # session has been closed, passing back to the while loop will stop the listener
                    continue
                
            except websocket.WebSocketTimeoutException:
                self.logger.warning("ws timeout exception occured in listener!")
                pass
            else:
                response = json.loads(msg)
                self.status.put(response)    
            
            """
            except Exception as e:
                #response = dict(messageType="ERROR", message=f"session failure, {e}")
                #self.status.put(response)
                self.logger.error("Error in listener! Exception: %s" %e)
                break
            
            # id resp_opcode != 8 ==> everything is OK, enqueue the message
            else:
                response = json.loads(msg)
                self.status.put(response)    
            """

        # use explicit return, otherwise the thread will not close
        return True    
    
    def __start_session(self, input="microphone", audiofile=None, outfile=None, keyListener=False, continue_session=False, ignore_ssl=False):
        """
        Start session by connecting to the web socket and starting microphone streamer, queue sender, queue listener and key listener. 
        All in their own threads. 
        """

        # STEP 1: open new ws connection and save session info
        self.logger.info("***************** new session *****************")
        self.logger.debug(f'USER: {self.TB_config["username"]}')
        self.logger.debug(f'IP address: {self.get_ip()}')

        # check if ignore_ssl option is set. If True, skip verification of the server's SSL certificate
        if ignore_ssl:
            sslopt={"cert_reqs": ssl.CERT_NONE}
        
        # check if proxy is set
        if self.TB_config.get('proxy', None):
            proxy = self.TB_config['proxy']
            self.logger.info(f"Using proxy: {proxy}")
        else:
            proxy=None

        self.logger.debug("opening websocket")
        self.session_start = time.time()
        
        try:
            if self.http_protocol == 'https':
                url_new_session = f'wss://{self.TB_config["ASR_server"]}:443/ws?access_token={self.token_attributes["access_token"]}'
                url_continue_session = f'wss://{self.TB_config["ASR_server"]}:443/ws?access_token={self.token_attributes["access_token"]}&session_id={self.last_sessionId}'
            else:
                url_new_session = f'ws://{self.TB_config["ASR_server"]}/ws?access_token={self.token_attributes["access_token"]}'
                url_continue_session = f'ws://{self.TB_config["ASR_server"]}/ws?access_token={self.token_attributes["access_token"]}&session_id={self.last_sessionId}'
             
            """
            There are three dimensions based on which the connection type is selected:
            1. session type: can be new session or a continuation of previously ended session.
            2. proxy: can be set to some proxy server or none
            3. SSL verification: can be required (default and recommended) or disabled
            """

            if continue_session:
                if not self.last_sessionId:
                    err_data = dict(error_ts=time.time(), error_type='CONTINUE-SESSION-ERROR', close_message={'id': str(uuid.uuid4()), 'message': "Client requested to continue last session, but last session unknown!"})
                    self.connection_errors.append(err_data)
                    self.logger.error(err_data)
                    return False
                elif proxy:
                    if ignore_ssl:
                        # connection for session continuation over proxy without SSL verification 
                        self.logger.debug(f"Establishing connection for session continuation over proxy without SSL verification")
                        self.session = websocket.create_connection(
                            url=url_continue_session, 
                            http_proxy_host=proxy.split(":")[0], 
                            http_proxy_port=proxy.split(":")[-1],
                            sslopt={"cert_reqs": ssl.CERT_NONE}
                            )
                    else:
                        # connection for session continuation over proxy with SSL verification
                        self.logger.debug(f"Establishing connection for session continuation over proxy with SSL verification")
                        self.session = websocket.create_connection(
                            url=url_continue_session, 
                            http_proxy_host=proxy.split(":")[0], 
                            http_proxy_port=proxy.split(":")[-1]
                            )
                else:
                    if ignore_ssl:
                        # connection for session continuation without SSL verification
                        self.logger.debug(f"Establishing connection for session continuation without SSL verification")
                        self.session = websocket.create_connection(
                            url=url_continue_session,
                            sslopt={"cert_reqs": ssl.CERT_NONE} 
                            )
                    else:
                        # connection for session continuation with SSL verification
                        self.logger.debug(f"Establishing connection for session continuation with SSL verification")
                        self.session = websocket.create_connection(
                            url=url_continue_session, 
                            )
            elif proxy:
                if ignore_ssl:
                    # connection for a new session over proxy without SSL verification
                    self.logger.debug(f"Establishing connection for a new session over proxy without SSL verification")
                    self.session = websocket.create_connection(
                        url=url_new_session, 
                        http_proxy_host=proxy.split(":")[0], 
                        http_proxy_port=proxy.split(":")[-1],
                        sslopt={"cert_reqs": ssl.CERT_NONE}
                        )
                else:
                    # connection for a new session over proxy with SSL verification
                    self.logger.debug(f"Establishing connection for a new session over proxy with SSL verification")
                    self.session = websocket.create_connection(
                        url=url_new_session, 
                        http_proxy_host=proxy.split(":")[0], 
                        http_proxy_port=proxy.split(":")[-1]
                        )
            else:
                if ignore_ssl:
                    # connection for a new session without SSL verification
                    self.logger.debug(f"Establishing connection for a new session without SSL verification")
                    self.session = websocket.create_connection(
                        url=url_new_session,
                        sslopt={"cert_reqs": ssl.CERT_NONE}
                        )
                else:
                    # connection for a new session with SSL verification
                    self.logger.debug(f"Establishing connection for a new session with SSL verification")
                    self.session = websocket.create_connection(
                        url=url_new_session
                        )

        except Exception as e:
            err_obj = TBError(error_code=4000, error_message="Error establishing websocket connection!", exception=e, error_type='CLIENT-ERROR')
            self.connection_errors.append(vars(err_obj))
            self.state = 'Error'
            return False

        self.logger.debug("websocket established")
        self.state = 'Connected'
        self.elapsed_time_to_connect = time.time() - self.session_start

        try:
            # read response from ws
            resp_opcode, msg = self.session.recv_data()
    
            # if resp_opcode == 8 --> websocket has been closed; see https://websocket-client.readthedocs.io/en/latest/examples.html
            if resp_opcode == 8 and len(msg) >= 2:
                self.__process_closed_ws_response(msg)
                return False
            # Response code doesn't show anything was wrong. Get returned information and check if an error is logged in it - this should happen in rare occasions
            # Otherwise, everything is ok, ws connected. Set last session ID
            else:
                sinfo=json.loads(msg)
                self.session_info.append(sinfo)
                self.last_sessionId = sinfo.get("sessionId", None)
                if 'messageType' in sinfo and sinfo['messageType']=='ERROR':
                    err_obj = TBError(error_code=4000, error_message=f"Handshake OK, but can't start session. {sinfo.get('message', None)}", error_type='CLIENT-ERROR')
                    self.connection_errors.append(var(err_obj))
                    return False
                self.logger.debug("session handshake ok")
                self.logger.info(f"session info: {sinfo}")

        # in case of any exception, return False
        except Exception as e:
            err_obj = TBError(error_code=4000, error_message="Error receiving data from the web socket for the handshake!", exception=e, error_type='CLIENT-ERROR')
            self.connection_errors.append(vars(err_obj))
            self.logger.error(vars(err_obj))
            self.state = 'Error'
            return False
        
            
        # STEP 2: ---------------------------------------------------------------------------------------------------------
        # At this point ws connection is established. Now start start a sender, listener, microphone and keyboard listener,
        # all in separated threads

        # start a sender
        try:
            self.sender = threading.Thread(target=self.__data_sender)
            self.sender.start()
            self.logger.debug("sender started")
        except Exception as e:
            err_obj = TBError(error_code=4000, error_message="Error starting sender in a thread!", exception=e, error_type='CLIENT-ERROR')
            self.connection_errors.append(vars(err_obj))
            self.logger.error(vars(err_obj))
            self.state = 'Error'
            return False
        
        # start a response listener
        try:
            self.listener = threading.Thread(target=self.__response_listener)
            self.listener.start()
            self.logger.debug("listener started")
        except Exception as e:
            err_obj = TBError(error_code=4000, error_message="Error starting listener in a thread!", exception=e, error_type='CLIENT-ERROR')
            self.connection_errors.append(vars(err_obj))
            self.logger.error(vars(err_obj))
            self.state = 'Error'
            return False

        # start microphone if input is set to "microphone" or start reading from the file
        if input=="microphone":
            if self.selected_input_device_index and self.selected_input_device_index != 0:
                self.logger.debug(f"Starting microphone stream using input device {self.input_devices[self.selected_input_device_index]}")
            else:
                self.logger.debug(f"Starting microphone stream using system default input device")
            try:
                self.sender = threading.Thread(target=self.__microphone_streamer)
                self.sender.start()
                self.logger.debug("microphone stream started")
            except Exception as e:
                err_obj = TBError(error_code=4000, error_message="Error starting microphone streamer in a thread!", exception=e, error_type='CLIENT-ERROR')
                self.connection_errors.append(vars(err_obj))
                self.logger.error(vars(err_obj))
                self.state = 'Error'
                return False
                
        elif input=="file":
            try:
                self.sender = threading.Thread(target=self.__enqueue_wave_frames, args=[audiofile])
                self.sender.start()
                self.logger.debug("file load to the queue started")
            except Exception as e:
                err_obj = TBError(error_code=4000, error_message="Error starting file streamer in a thread!", exception=e, error_type='CLIENT-ERROR')
                self.connection_errors.append(vars(err_obj))
                self.logger.error(vars(err_obj))
                self.state = 'Error'
                return False
            
        else:
            err_obj = TBError(error_code=4000, error_message="Wrong transcription input! Please use 'microphone' for transcribing microphone stream or 'file' if you want to transcribe an audio file.", exception=e, error_type='CLIENT-ERROR')
            self.connection_errors.append(vars(err_obj))
            self.logger.error(vars(err_obj))
            self.state = 'Error'
            return False

        # listen on keyboard for end session request
        if keyListener:
            listener = keyboard.Listener(on_release=self.__on_keyrelease)
            listener.start()
            self.logger.debug("key listener started")
        
        # Return true to the caller. But be aware, this response will be returned when all threads, i.e. the microphone streamer,
        # the listener and the sender (plus keyboard listener) are still running!
        return True

    def __process_closed_ws_response(self, msg):
        """
        Is called if ws was closed. It does the following:
        - unpacks close code and message. 
        - if close code defined in the ws_closing_codes list, it retrieves additional info
        - based on the close code it checks whether the closure was NORMAL or due to an ERROR
        - based on the closing reason it logs the event and if error, puts message to the session connection errors list
        See https://websocket-client.readthedocs.io/en/latest/examples.html for details on unpacking ws response
        """

        def extract_error_message(msg):
            try:
                message_struct = json.loads(msg)
                # if expected structure, return dict with keys id and message
                if 'id' in message_struct and 'message' in message_struct:
                    return message_struct
                else:
                    #if not in expected structure but is json string, return whatever you got
                    return dict(unstruct_message=message_struct)
                
            except json.JSONDecodeError:
                # return original message
                return dict(unstruct_message=msg)

        # unpack close code
        close_code = str(struct.unpack("!H", msg[0:2])[0])
        # close code is expected to be provided from the TB API and should be in int format and available in the ws_close_codes dictionary
        if close_code.isnumeric():
            close_code = int(close_code)
        # ws_close_codes dictionary differentiates between ERROR an INFO code. Mostly all codes are SERVER-ERROR - see attribute 'mtype'
        # if close_code is missing in the ws_close_codes dict, we consider it as an unknown error description
        if close_code not in self.ws_close_codes.keys():
            message_type = 'SERVER-ERROR'
            close_message_code = "UNKNOWN"
        # if close_code exists in the ws_close_codes and is of type 'ERROR' set the message_type to 'ERROR' otherwise 'INFO'
        else:
            message_type = self.ws_close_codes[close_code]['mtype']
            close_message_code = self.ws_close_codes[close_code]['status']
        # get the closing message that was sent over ws
        close_message = extract_error_message(msg[2:].decode("utf-8"))
        
        # put info about ws response to connection errors list if error happened
        if message_type[-5:] == 'ERROR':
            self.connection_errors.append(dict(
                error_ts=time.time(),
                error_type=message_type, 
                error_code=close_code, 
                error_id=close_message.get('id', '') if close_message else None,
                error_message=close_message.get('message', close_message.get('unstruct_message', '')) if close_message else None,
                )
            )
            self.logger.error("websocket closed with code: %s, message: %s" %(close_code, close_message_code))
            self.session = None
        else:
            self.logger.info("websocket closed with code: %s, message: %s" %(close_code, close_message_code))

    def end_session(self):
        """
        Public method to end a session
        """
        self.__end_session()

    def __end_session(self):
        """
        Private method to end a session
        """
        if self.session and self.session.connected:
            self.session.send_binary(b"")
        
            # wait until BE closes the session
            while self.session.connected:
                pass 

            self.logger.debug("session closed")
            self.session_info.append(dict(messageType='INFO', message='Session closed', duration='%0.1f seconds'%(time.time()-self.session_start)))
            self.connection_errors.clear()
        else:
            self.logger.warning("Calling end_session on closed web socket!")

        self.session = None
        self.state = 'Authenticated'

    def __on_keyrelease(self, key):
        """
        Private method to listen on keyboard for ESC or multimedia key PLAY/PAUSE to stop transcribing
        """
        if key==Key.media_play_pause:
            self.logger.debug('Multimedia key PLAY/PAUSE detected, stopping listener...')
            # stop listener
            self.__end_session()
            return False
        elif key == keyboard.Key.esc:
            self.logger.debug('\nESC key detected, stopping listener...')
            # stop listener
            self.__end_session()
            return False

    def get_status(self,block=False):
        """
        Public method to read transcripts, implemented as a generator. If everything is OK, response is returned. There are two exceptions that are 
        handled:
        - queueu.Empty - it occurs when we want to read from an empty queue - in this case we check is listener is working. If it works then this is
          ok and we pass is on. If listener is dead then something is wrong and we logge this and yield -1
        - general Exception - if any other exception occurs, we need to log it and yield -1
        NOTE: it is important tha the caller handles -1 as a potential return value.
        """
        try:
            response = self.status.get(block=block,timeout=self.timeout)
        except queue.Empty:
            # if it happens the Queue is empty, just go on unless the listener is dead
            if not self.listener.is_alive:
                err_obj = TBError(error_code=4000, error_message="Queue is empty and listener is dead!", exception=e, error_type='CLIENT-ERROR')
                self.logger.error(vars(err_obj))
                self.connection_errors.append(vars(err_obj))
                yield dict(status=-1)
            pass
        except Exception as e:
            # if error happens when reading the Queue (very unlikely), error will be logged but not propagated further, since we don't want to kill the web socket
            err_obj = TBError(error_code=4000, error_message="Error while reading from the response Queue!", exception=e, error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            self.connection_errors.append(vars(err_obj))
            yield dict(status=-1)
        else:
            yield response

    def upload_audio(self, filename, is_async):
        """
        Public method to upload an audio for transcription
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/upload?async={is_async}"
        
        files = {
            'file': ('filename', open(filename, 'rb')),
        }

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.post(url, headers=headers, files=files, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_session_status(self, sessionid, full_stack=False):
        """
        Public method to get session status after async upload call
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            if full_stack:
                return resp.json()
            else:
                return resp.json()['status']

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_session_data(self, sessionid):
        """
        Public method to read session data once finished
        TODO: rename to get_session_transcript
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/transcripts"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_session_metadata(self, sessionid):
        """
        Public method to get all available metadata on specific session
        TODO: rename to get_session_data
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_sessions(
        self,
        name=None,
        labels=[],
        createdAt=None,
        createdBy=None,
        source=None, 
        is_discarded=False,
        status='FINISHED', 
        page=0, 
        size=1, 
        sort='createdAt,desc',
        admin=False,
        ):
        """
        Public method to get all sessions from the authenticated user
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        # create filter
        if admin:
            filter = f"is-discarded={is_discarded}"
        else:
            filter = ""
        if name:
            filter += f"&name={name}"
        if createdAt:
            filter += f"&created-at={createdAt}"
        if createdBy:
            filter += f"&created-by-username={createdBy}"
        if source:
            filter += f"&source={source}"
        if status:
            filter += f"&status={status}"
        filter += f"&page={page}"
        filter += f"&size={size}"
        filter += f"&sort={sort}"

        for l in labels:
            filter += f"&label={l}"

        if filter[0]=='&':
            filter = filter[1:]

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions?{filter}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_client_labels(self):
        """
        Public method to get client labels
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers)

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def add_client_label(self, label_code, label_color, isDefault=False):
        """
        Public method to add new label for the client
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
        
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        json_data = {
            "code": label_code,
            "color": label_color,
            "isDefault": isDefault
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_session_labels(self, sessionid):
        """
        Public method to retrieve session labels
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def add_session_label(self, sessionid, labelid, is_enabled):
        """
        Public method to add session labels
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
        
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'labelId': labelid,
            'isEnabled': is_enabled,
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_user_replacements(self):
        """
        Public method to get user replacements
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/replacements"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
        
    def post_user_replacement(self, replacement):
        """
        Public method to post user replacements
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/replacements"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        json_data = replacement

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def delete_user_replacement(self, rid):
        """
        Public method to delete user replacement
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/replacements/{rid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.delete(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def patch_session_name(self, sessionid, name, notes=None):
        """
        Public method to patch session name
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
    	
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        if notes:
            json_data = {
                'name': name,
                'notes': notes,
            }
        else:
            json_data = {
                'name': name
            }

        resp = requests.patch(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
    
    def patch_session_label(self, sessionid, labelid, is_enabled):
        """
        Public method to patch session labels
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
        
    	#http://sandbox-api.true-bar.si/api/client/sessions/833/labels/2
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels/{labelid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'labelId': labelid,
            'isEnabled': is_enabled,
        }

        resp = requests.patch(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_users_in_my_group(self):
        """
        Get users in group
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/users"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    # TODO: check get_session_data() - possibly a duplicate function
    def get_session_transcript(self, sessionid, return_type='transcript'):
        """
        Returns transcript in TB format
        If you want to retrieve text only, use return_type=text
        To return list of finals, use return_type=finals
        TODO: this seems to be a duplicate of get_session_data!
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/transcripts"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            if return_type=='transcript' or return_type not in ('text', 'finals'):
                return resp.json()
            elif return_type=='text':
                transcript=""
                for ide in resp.json():
                    transcript_with_metadata = json.loads(ide["content"])
                    transcript += "".join([el["text"] if not el.get("spaceBefore", False) else " "+el["text"] for el in transcript_with_metadata])                
                return transcript
            elif return_type=='finals':
                finals = []
                for ide in resp.json():
                    transcript_with_metadata = json.loads(ide["content"])
                    finals.append("".join([el["text"] if not el.get("spaceBefore", False) else " "+el["text"] for el in transcript_with_metadata]))
                return finals
            else:
                self.logger.error(f"Unknown value for parameter 'return_type': {return_type}. Function 'get_session_transcript'")
                return False

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def add_session_shares(self, sessionid, userid):
        """
        Public method to add session share to a user
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False
    	
        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/shares"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'userId': userid,
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_session_audio(self, sessionid):
        """
        Public method to retreive session audio
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/audio.wav"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.content

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def play_audio(self, audiofile):
        """
        Public method to play audio
        """
        # open wave file
        wave_file = wave.open(audiofile, 'rb')
        
        # initialize audio
        py_audio = pyaudio.PyAudio()
        stream = py_audio.open(format=py_audio.get_format_from_width(wave_file.getsampwidth()),
                            channels=wave_file.getnchannels(),
                            rate=wave_file.getframerate(),
                            output=True)

        # skip unwanted frames
        #n_frames = int(start * wave_file.getframerate())
        #wave_file.setpos(n_frames)
        n_frames = wave_file.getnframes()

        # write desired frames to audio buffer
        #n_frames = int(length * wave_file.getframerate())
        frames = wave_file.readframes(n_frames)
        stream.write(frames)

        # close and terminate everything properly
        stream.close()
        py_audio.terminate()
        wave_file.close()

    def print_errorstack(self):
        """
        Public method to print error stack
        """
        for el in self.connection_errors:
            print(f"{el}\n") 

    def upload_logs(self, log_folder, host, username='marko'):
        """
        This is a custom procedure to upload log files and settings.yaml from a certain directory to the server.
        Files are first copied and renamed to fit destination format, i.e. <host>_<name>_<timestamp>_<filename>.{log|yaml}
        Notice that temp files are deleted at the beginning if there are any from the previous upload.
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"https://{host}-api.true-bar.si/logs/upload"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        counter = 0
        failed = 0

        # remove previous temp files if exist in the log folder
        for file in os.listdir(log_folder):
            if file[0]=="_":
                os.remove(os.path.join(log_folder, file))
        
        for file in os.listdir(log_folder):
            if file[-4:].lower() in ('.log', 'yaml'):
                counter += 1
                # make a copy of the log file and rename it to fir the destination format
                current_time_string = time.strftime("%Y%m%d_%H%M%S", time.localtime())
                destination_filename = os.path.join(log_folder, f"_{host}_{username}_{current_time_string}_{file}")

                shutil.copyfile(
                    os.path.join(log_folder, file), 
                    destination_filename
                    )
                files = {'file': open(destination_filename, 'rb')}

                resp = requests.post(
                    url=url, 
                    headers=headers, 
                    files=files,
                    verify=False
                    )
                
                try:
                    resp.raise_for_status()
                    self.logger.debug(f"Log file {file} uploaded.")

                except requests.exceptions.HTTPError as err:
                    failed += 1
                    err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
                    self.logger.error(vars(err_obj))

        return dict(all_log_files=counter, failed=failed)

    ### working with dictionary 
    def get_model_phoneset(self, language, domain, model):
        """
        Public method to get phoneset
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/phone-set"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, exception='HTTPError', error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

    def get_frequency_classes(self):
        """
        Public method to read frequency classes for the dictionary
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/frequencyClasses"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def read_dictionary(self, language, domain, model, token):
        """
        Public method to get all data for specific word
        """
        #https://demo-api.true-bar.si/api/client/dictionary/languages/sl-SI/domains/MED/dmodels/20221201-1735/words?search-by=word&text=ohranjena 
        #[{"id":10477891,"frequencyClassId":3,"status":"IN_DICTIONARY","text":"ohranjena","pronunciations":[{"saved":true,"text":"ohranjena"}]}]
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words?search-by=word&text={token}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def save_dictionary(self, language, domain, model, word_entry):
        """
        Public method to save word entry to dictionary
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = word_entry
        
        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_model_info(self, language, domain, model):
        """
        Public method to get model info
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
    
    def get_model_stat(self, language, domain, model):
        """
        Public method to get model statistics
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/stats"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def get_model_word_count(self, language, domain, model):
        """
        Public method to get model words count
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words/count"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def start_model_update(self, language, domain, model):
        """
        Public method to start model update
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/regeneration/regenerate"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.post(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def check_model_update_status(self, language, domain, model):
        """
        Public method to check model update status
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/regeneration/status"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
    
    ### working with personal dict for word boosting
    def get_user_boosting_words(self):
        """
        Public method to get user boosting records
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/boosted-phrases"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
        
    def post_user_boosting_words(self, boosting_item):
        """
        Public method to post user boosting record
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/boosted-phrases"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        json_data = boosting_item

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def delete_user_boosting_words(self, rid):
        """
        Public method to delete user boosting word
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/boosted-phrases/{rid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.delete(url, headers=headers, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def patch_user_boosting_words(self, rid, boosting_item):
        """
        Public method to patch user boosting record
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        url = f"{self.http_protocol}://"  + self.TB_config["ASR_server"] + f"/api/client/boosted-phrases/{rid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = boosting_item

        resp = requests.patch(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        try:
            resp.raise_for_status()
            return True

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    ### check web socket connection
    def check_connection(self, asr_server):
        """
        A public ping-pong method that establishes a web socket connection and retrieves info message from the server
        to confirm web socket connection is working  
        """

        def on_message(ws, message):
            self.logger.debug(f"Info Package received: {message}")
            ws.successful_connection = True
            ws.close()
            
        def on_error(ws, error):
            self.logger.error(f"Error on establishing web-socket test connection: {error}")

        def on_close(ws, close_status_code, close_msg):
            self.logger.debug(f"Connection closed with status code {close_status_code}")

        def on_open(ws):
            self.logger.debug("Connection established. Sending request...")

        #websocket.enableTrace(True)
    
        if asr_server[:7] != 'truebar':
            ws_url = f'wss://{asr_server}/api/common/ws-info'
        else:
            ws_url = f'ws://{asr_server}/api/common/ws-info'
        
        ws = websocket.WebSocketApp(
            ws_url,
            on_message=on_message,
            on_error=on_error,
            on_close=on_close
        )

        successful_connection = False

        ws.on_open = on_open
        ws.successful_connection = successful_connection
        ws.run_forever()

        if ws.successful_connection:
            return True
        else:
            return False

    ### get truecased text
    def truecase(self, text, tokenize=True, sessionid=None):
        """
        Temporary method to get truecased text
        """

        url = self.truecaser_url

        json_data = {
            'text': text,
            'tokenize': tokenize,
            'sessionid': sessionid,
        }
        
        resp = requests.post(url, json=json_data)

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    def truecase_tokens(self, tokens, sessionid=None):
        """Temporary method to truecase a list of tokens"""

        url = self.truecaser_tokens_url

        json_data = {
            'tokens': tokens,
            'sessionid': sessionid,
        }
        
        resp = requests.post(url, json=json_data)

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False

    ### get truecased text
    def truecase_safe(self, text, tokenize=True, sessionid=None):
        """
        Temporary method to get truecased text - with token verification
        """
        if not self.user_authenticated():
            err_obj = TBError(error_code=4000, error_message="User is not authenticated! Check the TruebarSTT method get_auth_token().", error_type='CLIENT-ERROR')
            self.logger.error(vars(err_obj))
            return False

        headers = CaseInsensitiveDict()
        headers["token"] = self.token_attributes["access_token"]
        headers["Content-Type"] = "application/json"

        url = self.truecaser_url

        json_data = {
            'text': text,
            'tokenize': tokenize,
            'sessionid': sessionid,
        }
        
        resp = requests.post(url, headers=headers, json=json_data)

        try:
            resp.raise_for_status()
            return resp.json()

        except requests.exceptions.HTTPError as err:
            err_obj = TBError(error_code=err.response.status_code, error_message=err.response.reason, error_type='CLIENT-ERROR', exception='HTTPError')
            self.logger.error(vars(err_obj))
            return False
        
    ### check if service on specific url is running
    def is_service_available(self, url):
        try:
            response = requests.head(url, timeout=1)  # Adjust the timeout as needed
            return True
        except requests.RequestException:
            return False
