from PyQt5 import QtCore, QtGui, QtWidgets
from collections import deque
import math


class RecordingOverlay(QtWidgets.QWidget):
    """
    A modern, fullscreen (per-screen) translucent overlay that visualizes active recording.

    Features
    --------
    - Pulsing radial ring + soft glow around a central mic button.
    - Minimal live audio level waveform (optional) via update_level(level: float).
    - Click/Tap anywhere to stop; Esc/Space/Enter also stop.
    - Smooth fade-in / fade-out transitions.
    - Always-on-top, frameless, translucent background.

    Signals
    -------
    stopRequested: emitted when the user clicks/taps the overlay or presses a stop key.

    Usage
    -----
    overlay = RecordingOverlay()
    overlay.show_overlay()              # fade in
    ... during recording ...
    overlay.update_level(rms_or_peak)   # 0..1 values, call at ~20–60 Hz (optional)
    connect overlay.stopRequested to your stop-recording logic.
    overlay.hide_overlay()              # fade out (called automatically on stopRequested unless you intercept)
    """

    stopRequested = QtCore.pyqtSignal()

    def __init__(self, parent=None, show_waveform=True, hint_text="Listening… Tap to stop"):
        super().__init__(parent)

        # Window flags & attributes for overlay behavior
        self.setWindowFlags(
            QtCore.Qt.FramelessWindowHint
            | QtCore.Qt.Tool               # stays out of taskbar, but on top
            | QtCore.Qt.WindowStaysOnTopHint
        )
        self.setAttribute(QtCore.Qt.WA_TranslucentBackground, True)
        self.setAttribute(QtCore.Qt.WA_ShowWithoutActivating, True)
        self.setMouseTracking(True)

        # State
        self._pulse = 0.0  # animated 0..1
        self._opacity_anim = None
        self._visible = False
        self._hint_text = hint_text
        self._show_waveform = show_waveform

        # Audio level buffer for simple waveform
        self._levels = deque([0.0] * 60, maxlen=120)  # start with a calm baseline

        # Timer to decay levels if updates pause
        self._decay_timer = QtCore.QTimer(self)
        self._decay_timer.setInterval(33)  # ~30 FPS
        self._decay_timer.timeout.connect(self._decay_levels)

        # Pulse animation (loop 0..1)
        self._pulse_anim = QtCore.QPropertyAnimation(self, b"pulse", self)
        self._pulse_anim.setStartValue(0.0)
        self._pulse_anim.setEndValue(1.0)
        self._pulse_anim.setDuration(1500)
        self._pulse_anim.setLoopCount(-1)
        self._pulse_anim.setEasingCurve(QtCore.QEasingCurve.InOutSine)

        # Opacity effect for fade in/out
        self._opacity_effect = QtWidgets.QGraphicsOpacityEffect(self)
        self._opacity_effect.setOpacity(0.0)
        self.setGraphicsEffect(self._opacity_effect)

        # Accessibility
        self.setAccessibleName("Recording overlay")
        self.setAccessibleDescription("Displays when recording is active. Click or press Esc/Space/Enter to stop.")

    # ---------- Public API ----------
    @QtCore.pyqtProperty(float)
    def pulse(self):
        return self._pulse

    @pulse.setter
    def pulse(self, value: float):
        self._pulse = float(value)
        self.update()

    def set_hint_text(self, text: str):
        self._hint_text = text
        self.update()

    def update_level(self, value: float):
        """Provide an audio level in [0, 1]. Call at ~30 Hz for smooth waveform."""
        v = 0.0 if value is None else max(0.0, min(1.0, float(value)))
        self._levels.append(v)
        if not self._decay_timer.isActive():
            self._decay_timer.start()
        self.update()

    def show_overlay(self, screen: QtGui.QScreen = None, fade_ms: int = 200):
        """Show the overlay on the given screen (or the screen under the cursor)."""
        if screen is None:
            screen = QtWidgets.QApplication.screenAt(QtGui.QCursor.pos()) or QtWidgets.QApplication.primaryScreen()
        if screen is None:
            # Fallback: normal show
            self.show()
        else:
            self.setGeometry(screen.geometry())
            # Use show() not showFullScreen() to avoid stealing focus on some WMs
            self.show()
        self.raise_()
        self._start_animations()
        self._fade_to(1.0, fade_ms)
        self._visible = True

    def hide_overlay(self, fade_ms: int = 150):
        if not self._visible:
            return
        self._fade_to(0.0, fade_ms, on_finished=self._finish_hide)
        self._visible = False

    # ---------- Events ----------
    def mousePressEvent(self, event: QtGui.QMouseEvent):
        if event.button() == QtCore.Qt.LeftButton:
            self._emit_stop()
        super().mousePressEvent(event)

    def keyPressEvent(self, event: QtGui.QKeyEvent):
        if event.key() in (QtCore.Qt.Key_Escape, QtCore.Qt.Key_Space, QtCore.Qt.Key_Return, QtCore.Qt.Key_Enter):
            self._emit_stop()
        else:
            super().keyPressEvent(event)

    def paintEvent(self, event: QtGui.QPaintEvent):
        p = QtGui.QPainter(self)
        p.setRenderHint(QtGui.QPainter.Antialiasing, True)

        rect = self.rect()
        w, h = rect.width(), rect.height()

        # Dimmed backdrop / scrim
        scrim = QtGui.QColor(10, 12, 16, 180)  # semi-opaque dark
        p.fillRect(rect, scrim)

        # Center geometry
        cx, cy = rect.center().x(), rect.center().y()
        base_radius = int(min(w, h) * 0.10)  # central button size ~20% of min dimension
        max_glow = int(base_radius * 1.8)

        # Pulsing glow ring
        pulse = self._pulse
        glow_radius = int(base_radius + (max_glow - base_radius) * (0.5 + 0.5 * math.sin(pulse * math.tau)))

        # Soft radial glow
        grad = QtGui.QRadialGradient(QtCore.QPointF(cx, cy), glow_radius)
        grad.setColorAt(0.0, QtGui.QColor(0, 200, 255, 90))
        grad.setColorAt(0.6, QtGui.QColor(0, 200, 255, 40))
        grad.setColorAt(1.0, QtGui.QColor(0, 200, 255, 0))
        p.setBrush(QtGui.QBrush(grad))
        p.setPen(QtCore.Qt.NoPen)
        p.drawEllipse(QtCore.QPointF(cx, cy), glow_radius, glow_radius)

        # Outer ring (subtle stroke that breathes)
        ring_alpha = 150 + int(80 * math.sin(pulse * math.tau))
        ring_pen = QtGui.QPen(QtGui.QColor(0, 200, 255, ring_alpha), max(2, base_radius * 0.08))
        p.setPen(ring_pen)
        p.setBrush(QtCore.Qt.NoBrush)
        p.drawEllipse(QtCore.QPointF(cx, cy), base_radius * 1.2, base_radius * 1.2)

        # Central circular button
        inner = QtGui.QColor(0, 180, 255)
        inner.setAlpha(230)
        p.setPen(QtCore.Qt.NoPen)
        p.setBrush(inner)
        p.drawEllipse(QtCore.QPointF(cx, cy), base_radius, base_radius)

        # Mic glyph
        self._draw_mic_icon(p, QtCore.QPointF(cx, cy), base_radius * 0.55)

        # Hint text
        self._draw_hint_text(p, cx, cy + int(base_radius * 1.55))

        # Waveform (optional)
        if self._show_waveform:
            self._draw_waveform(p, rect, cy + int(base_radius * 2.4))

        p.end()

    # ---------- Drawing helpers ----------
    def _draw_mic_icon(self, p: QtGui.QPainter, center: QtCore.QPointF, size: float):
        # Simple microphone shape using paths
        path = QtGui.QPainterPath()
        r = size
        x, y = center.x(), center.y()

        # Capsule for mic body
        body = QtCore.QRectF(x - r * 0.45, y - r * 0.60, r * 0.9, r * 1.2)
        path.addRoundedRect(body, r * 0.45, r * 0.45)

        # Stem
        stem_h = r * 0.55
        stem_w = r * 0.16
        stem = QtCore.QRectF(x - stem_w / 2, y + r * 0.65, stem_w, stem_h)
        path.addRoundedRect(stem, stem_w / 2, stem_w / 2)

        # Base
        base = QtCore.QRectF(x - r * 0.55, y + r * 1.15, r * 1.1, r * 0.18)
        path.addRoundedRect(base, r * 0.09, r * 0.09)

        p.setBrush(QtGui.QColor(255, 255, 255))
        p.setPen(QtCore.Qt.NoPen)
        p.drawPath(path)

    def _draw_hint_text(self, p: QtGui.QPainter, cx: int, y: int):
        # Subtitle-like copy
        font = QtGui.QFont()
        font.setPointSizeF(max(14.0, self.height() * 0.022))
        font.setWeight(QtGui.QFont.Medium)
        p.setFont(font)

        metrics = QtGui.QFontMetrics(p.font())
        text = self._hint_text
        tw = metrics.width(text)
        th = metrics.height()

        # Text shadow
        shadow_path = QtGui.QPainterPath()
        shadow_path.addText(QtCore.QPointF(cx - tw / 2 + 1.5, y + th / 4 + 1.5), p.font(), text)
        p.setPen(QtCore.Qt.NoPen)
        p.setBrush(QtGui.QColor(0, 0, 0, 160))
        p.drawPath(shadow_path)

        # Text fill
        p.setPen(QtGui.QColor(240, 248, 255))
        p.drawText(QtCore.QPointF(cx - tw / 2.0, y + th / 4.0), text)


    def _draw_waveform(self, p: QtGui.QPainter, rect: QtCore.QRect, baseline_y: int):
        if not self._levels:
            return
        margin = int(rect.width() * 0.12)
        width = rect.width() - 2 * margin
        height = int(rect.height() * 0.12)
        top = baseline_y - height // 2

        # Background bar (very subtle)
        bg_rect = QtCore.QRect(margin, top, width, height)
        p.setPen(QtCore.Qt.NoPen)
        p.setBrush(QtGui.QColor(255, 255, 255, 25))
        p.drawRoundedRect(bg_rect, height / 2, height / 2)

        # Build polyline based on levels (mirror around center for filled shape)
        n = min(len(self._levels), width)
        if n < 2:
            return
        samples = list(self._levels)[-n:]

        path = QtGui.QPainterPath()
        step = width / float(n - 1)
        cx = margin
        mid = top + height / 2
        amp = height * 0.48

        # Top outline
        path.moveTo(cx, mid)
        for i, v in enumerate(samples):
            y = mid - (v ** 0.6) * amp  # slightly emphasize low levels
            path.lineTo(margin + i * step, y)

        # Bottom outline (reverse)
        for i, v in reversed(list(enumerate(samples))):
            y = mid + (v ** 0.6) * amp
            path.lineTo(margin + i * step, y)
        path.closeSubpath()

        # Fill & stroke
        p.setBrush(QtGui.QColor(0, 200, 255, 120))
        p.setPen(QtGui.QPen(QtGui.QColor(255, 255, 255, 80), 1.2))
        p.drawPath(path)

    # ---------- Animation / internals ----------
    def _fade_to(self, target_opacity: float, ms: int, on_finished=None):
        anim = QtCore.QPropertyAnimation(self._opacity_effect, b"opacity", self)
        anim.setStartValue(self._opacity_effect.opacity())
        anim.setEndValue(target_opacity)
        anim.setDuration(ms)
        anim.setEasingCurve(QtCore.QEasingCurve.InOutQuad)
        if on_finished:
            anim.finished.connect(on_finished)
        anim.start(QtCore.QAbstractAnimation.DeleteWhenStopped)
        self._opacity_anim = anim

    def _finish_hide(self):
        self._stop_animations()
        self.hide()

    def _start_animations(self):
        if self._pulse_anim.state() != QtCore.QAbstractAnimation.Running:
            self._pulse_anim.start()

    def _stop_animations(self):
        if self._pulse_anim.state() == QtCore.QAbstractAnimation.Running:
            self._pulse_anim.stop()
        self._decay_timer.stop()

    def _decay_levels(self):
        # Gentle decay so the waveform settles when no updates are incoming
        if not self._levels:
            return
        decayed = deque(maxlen=self._levels.maxlen)
        any_nonzero = False
        for v in self._levels:
            nv = max(0.0, v - 0.02)
            any_nonzero = any_nonzero or nv > 0.001
            decayed.append(nv)
        self._levels = decayed
        if not any_nonzero:
            self._decay_timer.stop()
        self.update()

    def _emit_stop(self):
        # Immediate visual acknowledgement: brief flash + fade out
        self.stopRequested.emit()
        self.hide_overlay()


# ------------------- Demo harness -------------------
# Run this file directly to preview the overlay.
if __name__ == "__main__":
    def _demo():
        import random
        app = QtWidgets.QApplication([])

        overlay = RecordingOverlay(show_waveform=True)
        overlay.show_overlay()

        level_timer = QtCore.QTimer()
        level_timer.setInterval(33)

        level = 0.0
        direction = 1

        def feed_level():
            nonlocal level, direction
            if random.random() < 0.05:
                direction *= -1
            level += direction * 0.03
            level = max(0.0, min(1.0, level))
            if random.random() < 0.06:
                level = min(1.0, level + random.random() * 0.5)
            overlay.update_level(level)

        level_timer.timeout.connect(feed_level)
        level_timer.start()

        overlay.stopRequested.connect(lambda: print("stopRequested() emitted"))
        app.exec_()

    _demo()

