# note_preview_threepart.py
from __future__ import annotations
import json, re, html
from typing import Any, Dict, List, Optional, Tuple

from PyQt5.QtCore import Qt, QTimer, QElapsedTimer
from PyQt5.QtGui import QTextCursor, QTextCharFormat, QColor, QTextDocument, QFont
from PyQt5.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QToolBar, QAction, QLineEdit, QLabel,
    QSplitter, QTextEdit, QPlainTextEdit, QFileDialog, QMessageBox, QProgressBar
)
from PyQt5.QtPrintSupport import QPrinter


STRUCTURED_JSON_TAG = ("<STRUCTURED_JSON>", "</STRUCTURED_JSON>")
MARKDOWN_NOTE_TAG   = ("<MARKDOWN_NOTE>", "</MARKDOWN_NOTE>")
ISSUE_PH            = "<<<ISSUE-{n}>>>"

# Promote common Slovenian section titles, normalize bullets, fix code fences, etc.
_SECTION_TITLES = {
    "pregled", "anamneza", "status", "dg", "diagnoza",
    "terapija", "zdravljenje", "izvid", "mnenje", "zaključek", "priporočila",
    "ultrazvok trebuha - rezultati", "rezultati", "komentar"
}

# ---------- Parsing helpers ----------
def _extract_between(text: str, start_tag: str, end_tag: str) -> Optional[str]:
    m = re.search(re.escape(start_tag) + r"(.*?)" + re.escape(end_tag), text, flags=re.DOTALL | re.IGNORECASE)
    return m.group(1).strip() if m else None

def parse_llm_reply(reply_text: str) -> Tuple[Optional[Dict[str, Any]], str]:
    """
    Returns (json_obj_or_None, markdown_string).
    Supports tagged dual output and fenced ```json blocks.
    """
    raw_json = _extract_between(reply_text, *STRUCTURED_JSON_TAG)
    raw_md   = _extract_between(reply_text, *MARKDOWN_NOTE_TAG)

    js = None
    if raw_json:
        try:
            js = json.loads(raw_json)
        except Exception:
            try:
                js = json.loads(re.sub(r",(\s*[}\]])", r"\1", raw_json))
            except Exception:
                js = None

    md = (raw_md or "").strip() or reply_text.strip()

    if js is None:
        m = re.search(r"```json\s*(\{[\s\S]*?\})\s*```", reply_text, flags=re.IGNORECASE)
        if m:
            try:
                js = json.loads(m.group(1))
            except Exception:
                try:
                    js = json.loads(re.sub(r",(\s*[}\]])", r"\1", m.group(1)))
                except Exception:
                    js = None
    return js, md

# ---------- Editor notes extraction ----------
def extract_inline_issues(md: str) -> Tuple[str, List[Dict[str, str]]]:
    """Replace [nejasno: ...] / [preveri: ...] with placeholders and collect issues."""
    issues: List[Dict[str, str]] = []
    def repl(m):
        idx  = len(issues) + 1
        kind = m.group(1).lower()
        text = m.group(2).strip()
        issues.append({"id": f"issue-{idx}", "type": kind, "text": text})
        return ISSUE_PH.format(n=idx)
    md2 = re.sub(r"\[(nejasno|preveri)\s*:\s*(.+?)\]", repl, md, flags=re.IGNORECASE | re.DOTALL)
    return md2, issues

def remove_editors_section(md: str) -> Tuple[str, List[str]]:
    """Remove 'Opombe urednika' section and return bullet items."""
    pat_head = re.compile(r"(?im)^(?:#{1,6}\s+)?(?:Opombe urednika|Editor(?:'|’)s Notes)\s*$")
    m = pat_head.search(md)
    if not m:
        return md, []
    after = md[m.end():]
    m2 = re.search(r"(?im)^\s*(?:#{1,6}\s+.+)$", after)  # next heading
    block = after[:m2.start()] if m2 else after
    bullets = [li.strip() for li in re.findall(r"(?m)^\s*[-\*]\s+(.*)$", block) if li.strip()]
    md_wo = md[:m.start()] + (after[m2.start():] if m2 else "")
    return md_wo.strip(), bullets

# ---------- Markdown <-> HTML ----------
def markdown_to_html(md: str) -> str:
    try:
        import markdown  # type: ignore
        return markdown.markdown(md, extensions=["extra", "sane_lists"])
    except Exception:
        esc = html.escape(md)
        esc = re.sub(r"(?m)^###\s+(.*)$", r"<h3>\1</h3>", esc)
        esc = re.sub(r"(?m)^##\s+(.*)$",  r"<h2>\1</h2>", esc)
        esc = re.sub(r"(?m)^#\s+(.*)$",   r"<h1>\1</h1>", esc)
        lines, in_ul = [], False
        for line in esc.splitlines():
            if re.match(r"^\s*[-\*]\s+", line):
                if not in_ul: lines.append("<ul>"); in_ul = True
                lines.append("<li>"+re.sub(r"^\s*[-\*]\s+","",line)+"</li>")
            else:
                if in_ul: lines.append("</ul>"); in_ul = False
                lines.append("<p>"+line+"</p>" if line.strip() else "")
        if in_ul: lines.append("</ul>")
        return "\n".join(lines)

def html_to_markdown_simple(html_text: str) -> str:
    """Minimal HTML->Markdown converter (headings, paragraphs, lists, bold/italic)."""
    html_text = re.sub(r"<br\s*/?>", "\n", html_text, flags=re.IGNORECASE)
    def repl_h(m):
        tag = m.group(1).lower()
        text = re.sub(r"<.*?>", "", m.group(2)).strip()
        pre = "# " if tag=="h1" else "## " if tag=="h2" else "### "
        return f"\n{pre}{text}\n\n"
    html_text = re.sub(r"(?is)<(h[1-3])>(.*?)</\1>", repl_h, html_text)
    html_text = re.sub(r"(?is)<(b|strong)>(.*?)</\1>", r"**\2**", html_text)
    html_text = re.sub(r"(?is)<(i|em)>(.*?)</\1>", r"*\2*", html_text)
    def repl_ul(m):
        items = re.findall(r"(?is)<li>(.*?)</li>", m.group(1))
        items = [re.sub(r"<.*?>", "", it).strip() for it in items]
        return "\n" + "\n".join(f"- {it}" for it in items if it) + "\n\n"
    html_text = re.sub(r"(?is)<ul>(.*?)</ul>", repl_ul, html_text)
    html_text = re.sub(r"(?is)<p>\s*(.*?)\s*</p>", r"\1\n\n", html_text)
    text = re.sub(r"(?is)<.*?>", "", html_text)
    return re.sub(r"\n{3,}", "\n\n", text).strip()

# ---------- Build styled content HTML with inline editor-note tags ----------
def build_content_html(md: str, js: Optional[Dict[str, Any]]) -> str:
    md1, issues  = extract_inline_issues(md)
    md2, bullets = remove_editors_section(md1)

    # --- 0) Pre-normalize mixed LLM output so markdown_to_html works reliably ---
    md2 = _normalize_markdown_for_medical(md2)

    # Convert to HTML
    html_note = markdown_to_html(md2)

    # --- 1) Replace inline ISSUE placeholders with styled tags ---
    def tag_html(kind: str, text: str) -> str:
        if str(kind).lower().startswith("preveri"):
            bg, fg, lbl = "#f8d7da", "#842029", "PREVERI"
        else:
            bg, fg, lbl = "#fff3cd", "#8a6d3b", "NEJASNO"
        return (f"<span style=\"background:{bg};color:{fg};border-radius:6px;"
                f"padding:2px 6px;margin:0 2px;font-weight:600;\">{lbl}: {html.escape(text)}</span>")

    for i, it in enumerate(issues, start=1):
        ph = ISSUE_PH.format(n=i)
        repl = tag_html(it.get("type",""), it.get("text",""))
        # Replace both escaped and raw placeholders, just in case
        html_note = html_note.replace(html.escape(ph), repl).replace(ph, repl)

    # --- 2) Top callout for non-inline notes (bullets + JSON issues) ---
    callouts: List[str] = []
    if bullets: callouts.extend(bullets)
    if js and isinstance(js.get("issues_to_verify"), list):
        for t in js["issues_to_verify"]:
            s = str(t).strip()
            if s: callouts.append(s)
    if callouts:
        items = "".join(f"<li>{html.escape(s)}</li>" for s in callouts)
        callout_html = (
            "<div style='border:1px dashed #e2e8f0;background:#f8fafc;border-radius:10px;padding:8px 10px;margin-bottom:10px'>"
            "<strong>⚠️ Opombe urednika</strong>"
            f"<ul style='margin:6px 0 0 18px'>{items}</ul></div>"
        )
        html_note = callout_html + html_note

    # --- 3) Medical readability polish on the produced HTML ---
    html_note = _polish_medical_html(html_note)

    # --- 4) Heading tweaks (your original styles, kept) ---
    html_note = (html_note
        .replace("<h1>", "<h1 style='font-size:18px;margin:10px 0 6px'>")
        .replace("<h2>", "<h2 style='font-size:16px;margin:12px 0 6px;border-left:3px solid #2563eb;padding-left:6px'>")
        .replace("<h3>", "<h3 style='font-size:15px;margin:8px 0 4px;color:#475569'>")
    )
    return html_note

def _normalize_markdown_for_medical(text: str) -> str:
    if not text:
        return ""

    # Normalize newlines and strip surrounding code fences
    s = text.replace("\r\n", "\n").replace("\r", "\n").strip()
    s = re.sub(r"^```(?:[a-zA-Z0-9_-]+)?\n", "", s)           # opening ```lang
    s = re.sub(r"\n```$", "", s)                              # closing ```

    # Unify bullets that models often use (•, 1), 1., -, *)
    s = re.sub(r"(?m)^\s*•\s+", "- ", s)
    s = re.sub(r"(?m)^\s*(\d+)\)\s+", r"\1. ", s)            # 1) -> 1.
    # Ensure a blank line before lists so Markdown parsers detect them
    s = re.sub(r"(?m)([^\n])\n(- |\d+\. )", r"\1\n\n\2", s)

    # Promote bold-only lines (with or without trailing colon) to headings
    # **Pregled:**   -> ## Pregled
    s = re.sub(r"(?m)^\s*\*\*\s*([^*:\n][^*\n]{0,120}?)\s*\*\*\s*:?\s*$",
               lambda m: f"## {m.group(1).strip()}", s)

    # Promote plain section title lines (even if not bold) to headings
    def _sectionize(m):
        name = m.group(1).strip()
        if name.lower() in _SECTION_TITLES:
            return f"## {name}"
        return m.group(0)
    s = re.sub(r"(?m)^\s*([A-Za-zČŠŽĐćčšžÀ-ÿ0-9][^:\n]{0,120}?)\s*:?\s*$", _sectionize, s)

    # Collapse >2 blank lines
    s = re.sub(r"\n{3,}", "\n\n", s)

    return s

def _polish_medical_html(html_note: str) -> str:
    h = html_note

    # 1) Label: value → <p><b>Label:</b> value</p>  (safe, bounded label)
    # Avoid grabbing entire sentences: label ≤ 50 chars, no '<' '>' etc. before colon.
    def _label_wrap(m):
        label = m.group(1).strip()
        value = m.group(2).strip()
        return f"<p><b>{label}:</b> {value}</p>"

    h = re.sub(
        r"<p>\s*([A-Za-zČŠŽĐćčšžÀ-ÿ0-9][^:<]{0,50}?)\s*:\s*(.+?)\s*</p>",
        _label_wrap,
        h,
        flags=re.DOTALL
    )

    # 2) Fallback: if any **bold** survived the Markdown parser, convert it now
    h = re.sub(r"\*\*(.+?)\*\*", r"<b>\1</b>", h)

    # 3) Tidy spacing & list readability
    # - slightly larger line-height in paragraphs
    h = h.replace("<p>", "<p style='margin:0.35em 0;line-height:1.35;'>")
    # - list spacing
    h = h.replace("<ul>", "<ul style='margin:0.4em 0 0.4em 1.2em;'>")
    h = h.replace("<ol>", "<ol style='margin:0.4em 0 0.4em 1.2em;'>")
    h = h.replace("<li>", "<li style='margin:0.15em 0;'>")

    # 4) Normalize <strong> to <b> (so your heading CSS + weight look consistent)
    h = h.replace("<strong>", "<b>").replace("</strong>", "</b>")

    return h

# ---------- Main window ----------
class ClinicalNote(QMainWindow):
    """
    Three-pane preview (JSON | CONTENT | TRANSCRIPT) with global search.
    - Content is editable (QTextEdit).
    - JSON + TRANSCRIPT panels are hidden initially (collapsed + handles hidden).
    - Toolbar: toggle JSON/Transcript, Focus content, export Content (HTML/MD/TXT/DOCX), export JSON, copy Content.
    """

    def __init__(self, llm_reply_text: str, raw_transcript: str, parent=None, *, streaming: bool=False):
        super().__init__(parent)
        self.setWindowTitle("🧾 Predogled kliničnega zapisa")

        # runtime state
        self._elapsed_timer = QElapsedTimer()
        self._tick = QTimer(self); self._tick.setInterval(250); self._tick.timeout.connect(self._on_tick)
        self._cancel_cb = None

        # Parse LLM reply
        self.js, md = parse_llm_reply(llm_reply_text)

        # Build UI
        self._build_ui()
        self._build_status_bar()

        # Load panes
        self.json_view.setPlainText(json.dumps(self.js, ensure_ascii=False, indent=2) if self.js else "Ni JSON podatkov.")
        self.content_editor.setHtml(build_content_html(md, self.js))
        self.transcript_view.setPlainText(raw_transcript or "")

        # Start with side panels fully hidden (and handles hidden)
        self._center_only()
        self._center_with_margins()

        # Styling
        self._apply_styles()

        # if we plan to stream now, show the Transcript pane and set busy
        if streaming:
            self.set_busy(True, "Obdelujem transkript …")

    # ----- geometry -----
    def _center_with_margins(self):
        """
        Start in single-pane mode (center only).
        Size the window so that, if later expanded to 3 panes, it still fits inside the
        available screen margins. Then center based on the available screen rect.
        """
        screen = (self.windowHandle().screen().availableGeometry()
                if self.windowHandle() and self.windowHandle().screen()
                else QApplication.primaryScreen().availableGeometry())
        W, H = screen.width(), screen.height()

        margin_ratio = 0.12 if W >= 1920 else 0.08

        # Choose the width of ONE pane (center). 3 panes will be ~W*(1-2*margin).
        single_w = max(600, int(W * (1 - 2 * margin_ratio) / 3))
        single_h = max(600, int(H * (1 - 3 * margin_ratio)))

        self.resize(single_w, single_h)

        # Center the window using the available geometry
        x = screen.x() + (W - single_w) // 2
        y = screen.y() + (H - single_h) // 2
        self.move(x, y)


    # ----- UI build -----
    def _build_ui(self):
        central = QWidget(self); self.setCentralWidget(central)
        root = QVBoxLayout(central); root.setContentsMargins(10, 8, 10, 10); root.setSpacing(8)

        # Toolbar
        tb = QToolBar("Dejanja", self); self.addToolBar(tb)

        # Toggles
        act_focus        = QAction("Fokus: vsebina", self); act_focus.triggered.connect(self._center_only)
        act_toggle_left  = QAction("JSON", self);           act_toggle_left.triggered.connect(self._toggle_left)
        act_toggle_right = QAction("Transkript", self);     act_toggle_right.triggered.connect(self._toggle_right)

        # Search
        self.search_edit = QLineEdit(self)
        self.search_edit.setPlaceholderText("Iskanje v JSON / vsebini / transkriptu …")
        self.search_edit.textChanged.connect(self._highlight_all_panels)

        # Actions: copy / export
        act_copy     = QAction("Kopiraj vsebino", self); act_copy.triggered.connect(self.copy_content)
        act_exp_html = QAction("Izvozi HTML", self);     act_exp_html.triggered.connect(self.export_content_html)
        act_exp_md   = QAction("Izvozi MD", self);       act_exp_md.triggered.connect(self.export_content_md)
        act_exp_txt  = QAction("Izvozi TXT", self);      act_exp_txt.triggered.connect(self.export_content_txt)
        act_exp_docx = QAction("Izvozi DOCX", self);     act_exp_docx.triggered.connect(self.export_content_docx)
        act_exp_json = QAction("Izvozi JSON", self);     act_exp_json.triggered.connect(self.export_json)
        act_pdf      = QAction("PDF", self);             act_pdf.triggered.connect(self.print_pdf)

        # Compose toolbar
        tb.addAction(act_focus)
        tb.addAction(act_toggle_left)
        tb.addAction(act_toggle_right)
        tb.addSeparator()
        tb.addWidget(QLabel("🔎", self))
        tb.addWidget(self.search_edit)
        tb.addSeparator()
        for a in (act_copy, act_exp_html, act_exp_md, act_exp_txt, act_exp_docx, act_exp_json, act_pdf):
            tb.addAction(a)

        # Cancel action (hidden until busy)
        self.act_cancel = QAction("Prekliči", self)
        self.act_cancel.setVisible(False)
        self.act_cancel.triggered.connect(self._on_cancel_clicked)
        tb.addSeparator()
        tb.addAction(self.act_cancel)
        
        # Splitter with three panes
        self.splitter = QSplitter(self); self.splitter.setChildrenCollapsible(True)
        root.addWidget(self.splitter, 1)
        self._default_handle_width = self.splitter.handleWidth()  # remember for restore

        # Left: JSON (read-only)
        self.json_view = QPlainTextEdit(self); self.json_view.setReadOnly(True)
        self.splitter.addWidget(self.json_view)

        # Center: CONTENT (editable)
        self.content_editor = QTextEdit(self); self.content_editor.setAcceptRichText(True)
        self.splitter.addWidget(self.content_editor)

        # Right: TRANSCRIPT (read-only)
        self.transcript_view = QPlainTextEdit(self); self.transcript_view.setReadOnly(True)
        self.splitter.addWidget(self.transcript_view)

        # Initial proportion (center dominant; actual hiding done in _center_only)
        self.splitter.setSizes([240, 680, 240])

        # highlight selections storage
        self._sel_json = []; self._sel_content = []; self._sel_transcript = []

    # status bar widgets
    def _build_status_bar(self):
        sb = self.statusBar()
        self._status_label = QLabel("Pripravljeno", self)
        self._elapsed_label = QLabel("0.0 s", self)
        self._pbar = QProgressBar(self); self._pbar.setRange(0, 0); self._pbar.setVisible(False)
        sb.addWidget(self._status_label, 1)
        sb.addPermanentWidget(self._elapsed_label)
        sb.addPermanentWidget(self._pbar, 1)

    # public API to use with your worker
    def set_busy(self, on: bool, message: str = "Obdelujem …", cancel_cb=None):
        self._status_label.setText(message)
        self._pbar.setVisible(on)
        self.act_cancel.setVisible(on)
        self._cancel_cb = cancel_cb
        if on:
            self._elapsed_label.setText("0.0 s")
            self._elapsed_timer.start()
            self._tick.start()
        else:
            self._tick.stop()

    def _on_tick(self):
        ms = self._elapsed_timer.elapsed()
        self._elapsed_label.setText(f"{ms/1000.0:.1f} s")

    def _on_cancel_clicked(self):
        if self._cancel_cb:
            try: self._cancel_cb()
            except Exception: pass
        self._status_label.setText("Preklicano …")

    # streaming helpers
    def append_transcript_chunk(self, chunk: str):
        c = self.content_editor.textCursor()
        c.movePosition(c.End)
        c.insertText(chunk)
        self.content_editor.setTextCursor(c)

    def finalize_success(self, llm_reply_text: str, raw_transcript: str):
        """Call when streaming finishes OK."""
        try:
            self.js, md = parse_llm_reply(llm_reply_text)
            self.json_view.setPlainText(json.dumps(self.js, ensure_ascii=False, indent=2) if self.js else "Ni JSON podatkov.")
            self.content_editor.setHtml(build_content_html(md, self.js))
            self.transcript_view.setPlainText(raw_transcript or self.transcript_view.toPlainText())
        finally:
            self.set_busy(False, message="")

    def finalize_canceled(self):
        self._status_label.setText("Preklicano")
        self.set_busy(False)

    def finalize_error(self, msg: str):
        self._status_label.setText("Napaka")
        QMessageBox.critical(self, "Napaka", msg)
        self.set_busy(False)

    # ----- styles -----
    def _apply_styles(self):
        # 1) Set fonts explicitly (this actually affects the text content)
        content_font = QFont("Inter", 10)       # center pane
        json_font     = QFont("Consolas", 10)   # left pane (monospace)
        transcript_ft = QFont("Inter", 10)      # right pane

        # Prefer antialiased rendering (optional)
        content_font.setStyleStrategy(QFont.PreferAntialias)
        transcript_ft.setStyleStrategy(QFont.PreferAntialias)

        # QTextEdit -> set on the DOCUMENT (affects content)
        self.content_editor.document().setDefaultFont(content_font)

        # QPlainTextEdit -> set on the WIDGET (affects content)
        self.json_view.setFont(json_font)
        self.transcript_view.setFont(transcript_ft)

        # 2) Keep light borders/padding via stylesheet (visual chrome only)
        self.content_editor.setStyleSheet("""
            QTextEdit {
                background: #ffffff;
                border: 1px solid #e2e8f0;
                border-radius: 10px;
                padding: 12px;
                color: #0f172a;
            }
        """)
        self.json_view.setStyleSheet("""
            QPlainTextEdit {
                background: #fbfdff;
                border: 1px solid #e2e8f0;
                border-radius: 10px;
                padding: 8px;
                color: #0f172a;
            }
        """)
        self.transcript_view.setStyleSheet("""
            QPlainTextEdit {
                background: #ffffff;
                border: 1px solid #e2e8f0;
                border-radius: 10px;
                padding: 8px;
                color: #0f172a;
            }
        """)

    # ----- splitter handle visibility -----
    def _both_sides_hidden(self) -> bool:
        s = self.splitter.sizes()
        return (s[0] < 1) and (s[2] < 1)

    def _update_handle_visibility(self):
        """Hide handles and fully hide side widgets when both sides are collapsed."""
        if self._both_sides_hidden():
            self.splitter.setHandleWidth(0)
            self.json_view.setVisible(False)
            self.transcript_view.setVisible(False)
        else:
            self.splitter.setHandleWidth(self._default_handle_width)
            self.json_view.setVisible(True)
            self.transcript_view.setVisible(True)

    # ----- show/hide helpers -----
    def _center_only(self):
        """Collapse both sides. Keep center width; window width == center width; recenter."""
        s = self.splitter.sizes()
        center_w = s[1] if s[1] > 0 else max(300, self.content_editor.width() or self.splitter.size().width())
        target_total = center_w
        current_total = sum(s)
        delta = target_total - current_total
        if delta:
            self.resize(self.width() + int(delta), self.height())

        self.splitter.setSizes([0, center_w, 0])
        self._update_handle_visibility()
        self._recenter_horiz(left_w=0, center_w=center_w)

    def _toggle_left(self):
        s = self.splitter.sizes()
        if s[0] < 8: self._show_left()
        else:        self._hide_left()

    def _toggle_right(self):
        s = self.splitter.sizes()
        if s[2] < 8: self._show_right()
        else:        self._hide_right()

    def _hide_left(self):
        """Hide JSON (left). Keep center width; shrink window by one center width if needed; recenter."""
        s = self.splitter.sizes()
        if s[0] <= 0:
            return
        right_visible = s[2] > 0
        center_w = s[1] if s[1] > 0 else max(300, self.content_editor.width() or self.splitter.size().width())

        parts_after = 1 + (1 if right_visible else 0)  # center + maybe right
        target_total = center_w * parts_after
        current_total = sum(s)
        delta = target_total - current_total  # negative -> shrink
        if delta:
            self.resize(self.width() + int(delta), self.height())

        new_sizes = [0, center_w, (center_w if right_visible else 0)]
        self.splitter.setSizes(new_sizes)
        self._update_handle_visibility()

        self._recenter_horiz(left_w=0, center_w=center_w)

    def _hide_right(self):
        """Hide Transcript (right). Keep center width; shrink window by one center width if needed; recenter."""
        s = self.splitter.sizes()
        if s[2] <= 0:
            return
        left_visible = s[0] > 0
        center_w = s[1] if s[1] > 0 else max(300, self.content_editor.width() or self.splitter.size().width())

        parts_after = 1 + (1 if left_visible else 0)  # center + maybe left
        target_total = center_w * parts_after
        current_total = sum(s)
        delta = target_total - current_total  # negative -> shrink
        if delta:
            self.resize(self.width() + int(delta), self.height())

        new_sizes = [(center_w if left_visible else 0), center_w, 0]
        self.splitter.setSizes(new_sizes)
        self._update_handle_visibility()

        self._recenter_horiz(left_w=(center_w if left_visible else 0), center_w=center_w)

    def _recenter_horiz(self, left_w: int, center_w: int):
        """
        Move the window so the center pane's horizontal center aligns to the screen center.
        Keeps current Y; clamps X to the available screen area.
        """
        screen = (self.windowHandle().screen().availableGeometry()
                if self.windowHandle() and self.windowHandle().screen()
                else QApplication.primaryScreen().availableGeometry())
        screen_cx = screen.x() + screen.width() // 2

        new_x = int(screen_cx - (left_w + center_w / 2))
        # Clamp inside the available rect
        max_x = screen.right() - self.width()
        new_x = max(screen.x(), min(new_x, max_x))

        self.move(new_x, self.y())
    
    def _show_left(self):
        """Show JSON (left). Keep center width; give left the SAME width; expand window; recenter."""
        s = self.splitter.sizes()
        left_visible  = s[0] > 0
        right_visible = s[2] > 0
        if left_visible:
            return

        center_w = s[1] if s[1] > 0 else max(300, self.content_editor.width() or self.splitter.size().width())
        parts_after = 1 + 1 + (1 if right_visible else 0)  # center + left + maybe right
        target_total = center_w * parts_after
        current_total = sum(s)
        delta = target_total - current_total
        if delta:
            self.resize(self.width() + int(delta), self.height())

        # Equal widths for visible panes
        self.json_view.setVisible(True)
        new_sizes = [center_w, center_w, (center_w if right_visible else 0)]
        self.splitter.setSizes(new_sizes)
        self._update_handle_visibility()

        # Recenter so center pane stays in screen middle
        self._recenter_horiz(left_w=center_w, center_w=center_w)

    def _show_right(self):
        """Show Transcript (right). Keep center width; give right the SAME width; expand window; recenter."""
        s = self.splitter.sizes()
        left_visible  = s[0] > 0
        right_visible = s[2] > 0
        if right_visible:
            return

        center_w = s[1] if s[1] > 0 else max(300, self.content_editor.width() or self.splitter.size().width())
        parts_after = 1 + 1 + (1 if left_visible else 0)  # center + right + maybe left
        target_total = center_w * parts_after
        current_total = sum(s)
        delta = target_total - current_total
        if delta:
            self.resize(self.width() + int(delta), self.height())

        self.transcript_view.setVisible(True)
        new_sizes = [(center_w if left_visible else 0), center_w, center_w]
        self.splitter.setSizes(new_sizes)
        self._update_handle_visibility()

        self._recenter_horiz(left_w=(center_w if left_visible else 0), center_w=center_w)

    # ----- global search / highlight -----
    def _clear_all_highlights(self):
        self._sel_json.clear();       self.json_view.setExtraSelections([])
        self._sel_content.clear();    self.content_editor.setExtraSelections([])
        self._sel_transcript.clear(); self.transcript_view.setExtraSelections([])

    def _highlight_panel(self, widget, q: str, case_sensitive: bool):
        """Highlight all matches in QTextEdit or QPlainTextEdit using QTextEdit.ExtraSelection."""
        if not q or len(q) < 2:
            widget.setExtraSelections([])
            return []

        doc = widget.document()
        cursor = QTextCursor(doc); cursor.movePosition(QTextCursor.Start)

        fmt = QTextCharFormat(); fmt.setBackground(QColor("#fde68a"))  # light amber
        flags = QTextDocument.FindFlags()
        if case_sensitive:
            flags |= QTextDocument.FindCaseSensitively

        selections = []
        while True:
            cursor = doc.find(q, cursor, flags)
            if cursor.isNull(): break
            sel = QTextEdit.ExtraSelection()
            sel.cursor = cursor
            sel.format = fmt
            selections.append(sel)

        widget.setExtraSelections(selections)
        return selections

    def _highlight_all_panels(self, q: str):
        self._clear_all_highlights()
        case_sens = any(ch.isupper() for ch in q)
        self._sel_json       = self._highlight_panel(self.json_view,       q, case_sens)
        self._sel_content    = self._highlight_panel(self.content_editor,  q, case_sens)
        self._sel_transcript = self._highlight_panel(self.transcript_view, q, case_sens)

    # ----- copy / export -----
    def copy_content(self):
        QApplication.clipboard().setText(self.content_editor.toPlainText())
        QMessageBox.information(self, "Kopirano", "Vsebina skopirana v odložišče (TXT).")

    def export_content_html(self):
        path, _ = QFileDialog.getSaveFileName(self, "Izvozi HTML", "vsebina.html", "HTML (*.html)")
        if not path: return
        with open(path, "w", encoding="utf-8") as f:
            f.write(self.content_editor.toHtml())
        QMessageBox.information(self, "Shranjeno", f"HTML shranjen v:\n{path}")

    def export_content_md(self):
        path, _ = QFileDialog.getSaveFileName(self, "Izvozi Markdown", "vsebina.md", "Markdown (*.md)")
        if not path: return
        md = html_to_markdown_simple(self.content_editor.toHtml())
        with open(path, "w", encoding="utf-8") as f:
            f.write(md)
        QMessageBox.information(self, "Shranjeno", f"Markdown shranjen v:\n{path}")

    def export_content_txt(self):
        path, _ = QFileDialog.getSaveFileName(self, "Izvozi TXT", "vsebina.txt", "Text (*.txt)")
        if not path: return
        with open(path, "w", encoding="utf-8") as f:
            f.write(self.content_editor.toPlainText())
        QMessageBox.information(self, "Shranjeno", f"TXT shranjen v:\n{path}")

    def export_content_docx(self):
        path, _ = QFileDialog.getSaveFileName(self, "Izvozi DOCX", "vsebina.docx", "DOCX (*.docx)")
        if not path: return
        try:
            from docx import Document  # python-docx
            from docx.shared import Pt
        except Exception:
            QMessageBox.critical(self, "Manjka knjižnica", "Za izvoz DOCX namestite paket 'python-docx'.\n\npip install python-docx")
            return
        doc = Document()
        doc.styles['Normal'].font.name = 'Segoe UI'
        doc.styles['Normal'].font.size = Pt(11)
        text = self.content_editor.toPlainText().strip()
        for block in re.split(r"\n{2,}", text):
            doc.add_paragraph(block)
        doc.save(path)
        QMessageBox.information(self, "Shranjeno", f"DOCX shranjen v:\n{path}")

    def export_json(self):
        if not self.js:
            QMessageBox.warning(self, "Ni podatkov", "V odgovoru ni JSON podatkov.")
            return
        path, _ = QFileDialog.getSaveFileName(self, "Izvozi JSON", "podatki.json", "JSON (*.json)")
        if not path: return
        with open(path, "w", encoding="utf-8") as f:
            json.dump(self.js, f, ensure_ascii=False, indent=2)
        QMessageBox.information(self, "Shranjeno", f"JSON shranjen v:\n{path}")

    def print_pdf(self):
        path, _ = QFileDialog.getSaveFileName(self, "Natisni v PDF", "klinicni_zapis.pdf", "PDF (*.pdf)")
        if not path: return
        printer = QPrinter(QPrinter.HighResolution)
        printer.setOutputFormat(QPrinter.PdfFormat)
        printer.setOutputFileName(path)
        self.content_editor.document().print_(printer)
        QMessageBox.information(self, "Shranjeno", f"PDF shranjen v:\n{path}")

# ---------- Demo ----------
if __name__ == "__main__":
    import sys
    demo_reply = """
<STRUCTURED_JSON>{
  "patient": {"age_years": 72, "sex": "female", "allergies": [{"substance":"penicilin","reaction":"izpuščaj"}]},
  "vitals": {"heart_rate_bpm": 88, "systolic_mmHg": 138, "diastolic_mmHg": 82, "temperature_C": 37.1, "spo2_percent": 96},
  "medications": [{"name":"metformin","strength_value":500,"strength_unit":"mg","dose":"1 tableta","route":"po","frequency":"2× dnevno","prn":false}],
  "diagnoses": [{"label":"akutni bronhitis","icd10": null}],
  "plan": ["Simptomatsko zdravljenje", "Kontrola čez 1 teden"],
  "studies": ["CRP danes 12 mg/L"],
  "history": {"past_medical":["sladkorna bolezen tip 2","hipertenzija"],"surgical":[]},
  "issues_to_verify": ["preveri: trajanje kašlja – ali 7 ali 10 dni?", "nejasno: kateri antitusični sirup je na voljo doma?"]
}</STRUCTURED_JSON>

<MARKDOWN_NOTE>
## Subjektivno
72-letna ženska prihaja zaradi produktivnega kašlja, ki se je začel pred približno tednom dni. [preveri: trajanje kašlja – ali 7 ali 10 dni?]
Zanika vročino, občasno občutek toplote ponoči. [nejasno: kateri antitusični sirup je na voljo doma?]

## Objektivno
RR 138/82, pulz 88/min, T 37,1 °C, SpO₂ 96 %. Os auskultatorno brez piskanjem.

## Ocena
Klinična slika skladna z akutnim bronhitisom.

## Načrt
- Simptomatsko zdravljenje, hidracija
- Kontrola čez 7 dni
</MARKDOWN_NOTE>
"""
    demo_transcript = "to je gospa stara dvainsedemdeset let z kašljem teden dni ... zanika vročino včasih se počuti vroče ..."
    app = QApplication(sys.argv)
    w = ClinicalNote(demo_reply, demo_transcript)
    w.show()
    sys.exit(app.exec_())
