"""
truebar.py is a supporting module for TrueBar speech technologies

TruebarSTT class encapsulates several supporting methods that facilitate the communication with the Truebar ASR service. 
The class is written to support demo examples that come together with the Truebar Demo platform. 

Author: VITASIS inc.
Date: May 2021

TODO:
[ ] Check and remove transcribe_from_file and transcribe_from_mic if redundant
[x] Implement logging
[ ] The library should never exit from application but rather get back to the caller whatever happened.
    It should return a message or an exception in case an error occured. Generic response should be a 
    dict with the following keys:
    - status: str: {'failed'|'success'}
    - status_code: int # if web service, status code as returned from the service otherwise None
    - message: str # whatever message you want to send back to the client.
    Appart from that, all important situations must be logged using logger.
[ ] After long time of inactivity, authentication becomes invalid! This is not handled!!!
"""

import websocket
import threading
import time
import requests
import json
from pynput.keyboard import Key, Listener
from pynput import keyboard
import queue
import pyaudio
from requests.structures import CaseInsensitiveDict
import codecs
import struct
import logging
from logging import config
import wave
import yaml
import os
import shutil
import re
from html2text import html2text
import socket
import certifi

#from PyQt5.QtWidgets import QTextEdit

# logger
def setup_logging(logger_name, logger_folder):
    # read logger config and make sure to use correct logger folder
    with open('./conf/config-logger.yaml', 'r', encoding="utf-8") as stream:
        cfg = yaml.load(stream, Loader=yaml.FullLoader)
    cfg['handlers']['info_file_handler']['filename'] = os.path.join(logger_folder, 'info.log')
    cfg['handlers']['debug_file_handler']['filename'] = os.path.join(logger_folder, 'debug.log')
    cfg['handlers']['error_file_handler']['filename'] = os.path.join(logger_folder, 'error.log')
    logging.config.dictConfig(cfg)
    return logging.getLogger(logger_name)


class TranscriptReader():
    def __init__(
        self, 
        allowed_commands=[],
        allowed_punctuations=[],
        ):

        # dictionary that holds current state of tags
        self.tag_state=dict()
        # a list of token elements, each element has a text, spaceBefore tag, start and end time ...
        self.transcript_tokens=[]

        # command to html
        self.command_html_translation = {
            '<b>': '<b>',
            '</b>': '</b>', 
            '<i>': '<i>',
            '</i>': '</i>',
            '<u>': '<u>',
            '</u>': '</u>',
            '<uc>': '<div style="text-transform: uppercase;">',
            '</uc>': '</div>',
            '<lc>': '<div style="text-transform: lowercase;">',
            '</lc>': '</div>',
            '<alignl>': '<p align="left">',
            '<alignr>': '<p align="right">',
            '<alignc>': '<p align="center">',
            '<nl>': '<br>',
            '<np>': '<br><br>',
        }

        # commands that are not allowed will be ignored
        if len(allowed_commands)>0:
            self.allowed_commands = allowed_commands
        else:
            self.allowed_commands=['<nl>', '<np>', '<b>', '</b>', '<i>', '</i>', '<u>', '</u>', '<alignl>', '<alignr>', '<alignc>', '<uc>', '</uc>', '<lc>', '</lc>', '<spc>']

        # punctuation commands that are not allowed will be ignored
        if len(allowed_punctuations)>0:
            self.allowed_punctuations = allowed_punctuations
        else:
            # TODO: check with the puct in database
            self.allowed_punctuations=['<.>', '<,>', '<!>', '<?>', '<:>', '<;>', '<->', '<%>', '<°>', '<+>', '<">', '<×>']

        # regex pattern for extracting punct or command
        self._cmd_ptr = re.compile(r"^(<)(.*)(>)$")

    def _command_to_html(self, command):
        """
        Returns html code that implements action of the command. 
        If no translation is found, empty string is returned
        """ 
        return self.command_html_translation.get(command['text'], "")
        
    def _punctuation_to_html(self, punctuation):
        """
        Returns html code that corresponds to the punctuation command
        """
        punct = re.sub(self._cmd_ptr, r"\g<2>", punctuation['text'])
        
        if punctuation.get('spaceBefore', False):
            return " "+punct
        else:
            return punct

    def html_to_text(self, html):
        """
        Exrats text from the html
        """
        return html2text(html)

    def _parse_transcript(self, transcript_with_metadata):
        """
        Takes transcript in the form as returned from the ASR Kaldi and
        filles data structures of the class 
        TODO: remove not needed metadata
        """
        self.transcript_tokens=[]
        for ide in transcript_with_metadata:
            content = json.loads(ide['content'])
            for el in content:
                self.transcript_tokens.append(el)

    def transcript_to_html(self, transcript):
        """
        Takes transcript in the form as returned from the ASR Kaldi and 
        returns corresponding html code.
        """
        self._parse_transcript(transcript)
        
        text = ""
        for cdx, t in enumerate(self.transcript_tokens):
            if t['text'] in self.allowed_punctuations:
                text += self._punctuation_to_html(t)
            elif t['text'] in self.allowed_commands:
                text += self._command_to_html(t)
            elif re.match(self._cmd_ptr, t['text']):
                pass
            else:
                #if cdx > 0: print(f"text={t['text']}, spaceBefore={t['spaceBefore']}, previous_token={self.transcript_tokens[cdx-1]}, {t['text'].isnumeric()}, {self.transcript_tokens[cdx-1]['text'].isnumeric()}")
                if t['spaceBefore'] or \
                    (cdx > 0 and f"<{self.transcript_tokens[cdx-1]['text']}>" in self.allowed_punctuations and \
                        (not t['text'].isnumeric() or not self.transcript_tokens[cdx-1]['text'].isnumeric())
                    ):
                    text += " " + t['text']
                else:
                    text += t['text']
            
        return text

class TranscriptAligner():
    def __init__(
        self, 
        content, 
        transcript, 
        logger_folder=os.path.dirname(__file__)
        ):
        self.content = content
        self.transcript = transcript
        # updatetranscript with absolute positions of tokens in the text (content)
        self._update_transcript_with_pos()
        # multiple space pattern
        self._MULTIPLE_SPACES_PTRN = re.compile(r'\s{2,}')
        # set logger
        self.logger = setup_logging("TB_aligner", logger_folder)

    def _update_transcript_with_pos(self):
        """
        Updates the transcript list with absolute positions of tokens in the content
        Args:
        - None
        """
        current_position = 0
        for c, el in enumerate(self.transcript):
            position_in_text = self.content.find(el['text'], current_position)
            el.update(pos=position_in_text)
            if position_in_text != -1:
                current_position = position_in_text

    def _remove_punctuations(
        self,
        text,
        punctuations = ".,!?:;" 
        ):
        """
        This will remove all punctuations, even dot symbols after abbreviations.
        To deal with abbreviations differently, use 'remove_punctuations' method 
        from the Vitasis.Extractor class!
        """
        return re.sub(rf"[{punctuations}]", "", text)

    def _remove_multiple_spaces(self, t):
        """
        Converts multiple into a single space
        """
        return re.sub(self._MULTIPLE_SPACES_PTRN, " ", t)

    def align_text(
        self, 
        text1, 
        text2, 
        convert_line_breaks=True,
        remove_punctuation=True, 
        lowercase=True,
        ):
        """
        Aligns tokens from text1 and text2. Returns results where both texts are 
        represented with a list of their tokens except in places where the alignment
        was not posible. In such places "-" char is used to indicate that at that
        position alignment was not established.
        Args:
        - text1: first text
        - text2: second text
        - convert_line_breaks: if True, line breaks \n will be converted to a single space
        - remove_punctuation: if True, punctuations are removed
        - lowercase: if True, texts are first lowercased
        """
        match_award      = 20
        mismatch_penalty = -5
        gap_penalty      = -5 # both for opening and extanding

        def zeros(shape):
            retval = []
            for x in range(shape[0]):
                retval.append([])
                for y in range(shape[1]):
                    retval[-1].append(0)
            return retval

        def match_score(alpha, beta):
            if alpha == beta:
                return match_award
            elif alpha == '-' or beta == '-':
                return gap_penalty
            else:
                return mismatch_penalty

        def finalize(align1, align2):
            align1.reverse()    #reverse sequence 1
            align2.reverse()    #reverse sequence 2
            
            i,j = 0,0
            
            #calcuate identity, score and aligned sequeces

            found = 0
            score = 0
            identity = 0
            for i in range(0,len(align1)):
                # if two AAs are the same, then output the letter
                if align1[i] == align2[i]:                

                    identity = identity + 1
                    score += match_score(align1[i], align2[i])
            
                # if they are not identical and none of them is gap
                elif align1[i] != align2[i] and align1[i] != '-' and align2[i] != '-': 
                    score += match_score(align1[i], align2[i])
                    found = 0
            
                #if one of them is a gap, output a space
                elif align1[i] == '-' or align2[i] == '-':          
                    score += gap_penalty
            
            identity = float(identity) / len(align1)
            
            return identity, score, align1, align2

        def needle(seq1, seq2):
            m, n = len(seq1), len(seq2)  # length of two sequences
            
            # Generate DP table and traceback path pointer matrix
            score = zeros((m+1, n+1))      # the DP table
        
            # Calculate DP table
            for i in range(0, m + 1):
                score[i][0] = gap_penalty * i
            for j in range(0, n + 1):
                score[0][j] = gap_penalty * j
            for i in range(1, m + 1):
                for j in range(1, n + 1):
                    match = score[i - 1][j - 1] + match_score(seq1[i-1], seq2[j-1])
                    delete = score[i - 1][j] + gap_penalty
                    insert = score[i][j - 1] + gap_penalty
                    score[i][j] = max(match, delete, insert)

            # Traceback and compute the alignment 
            align1, align2 = [], []
            i,j = m,n # start from the bottom right cell
            while i > 0 and j > 0: # end toching the top or the left edge
                score_current = score[i][j]
                score_diagonal = score[i-1][j-1]
                score_up = score[i][j-1]
                score_left = score[i-1][j]

                if score_current == score_diagonal + match_score(seq1[i-1], seq2[j-1]):
                    align1.append(seq1[i-1])
                    align2.append(seq2[j-1])
                    i -= 1
                    j -= 1
                elif score_current == score_left + gap_penalty:
                    align1.append(seq1[i-1])
                    align2.append('-')
                    i -= 1
                elif score_current == score_up + gap_penalty:
                    align1.append('-')
                    align2.append(seq2[j-1])
                    j -= 1

            # Finish tracing up to the top left cell
            while i > 0:
                align1.append(seq1[i-1])
                align2.append('-')
                i -= 1
            while j > 0:
                align1.append('-')
                align2.append(seq2[j-1])
                j -= 1

            return finalize(align1, align2)

        def water(seq1, seq2):
            m, n = len(seq1), len(seq2)  # length of two sequences
            
            # Generate DP table and traceback path pointer matrix
            score = zeros((m+1, n+1))      # the DP table
            pointer = zeros((m+1, n+1))    # to store the traceback path
            
            max_score = 0        # initial maximum score in DP table
            # Calculate DP table and mark pointers
            for i in range(1, m + 1):
                for j in range(1, n + 1):
                    score_diagonal = score[i-1][j-1] + match_score(seq1[i-1], seq2[j-1])
                    score_up = score[i][j-1] + gap_penalty
                    score_left = score[i-1][j] + gap_penalty
                    score[i][j] = max(0,score_left, score_up, score_diagonal)
                    if score[i][j] == 0:
                        pointer[i][j] = 0 # 0 means end of the path
                    if score[i][j] == score_left:
                        pointer[i][j] = 1 # 1 means trace up
                    if score[i][j] == score_up:
                        pointer[i][j] = 2 # 2 means trace left
                    if score[i][j] == score_diagonal:
                        pointer[i][j] = 3 # 3 means trace diagonal
                    if score[i][j] >= max_score:
                        max_i = i
                        max_j = j
                        max_score = score[i][j];
            
            align1, align2 = [], []    # initial sequences
            
            i,j = max_i,max_j    # indices of path starting point
            
            #traceback, follow pointers
            while pointer[i][j] != 0:
                if pointer[i][j] == 3:
                    align1.append(seq1[i-1])
                    align2.append(seq2[j-1])
                    i -= 1
                    j -= 1
                elif pointer[i][j] == 2:
                    align1.append('-')
                    align2.append(seq2[j-1])
                    j -= 1
                elif pointer[i][j] == 1:
                    align1.append(seq1[i-1])
                    align2.append('-')
                    i -= 1

            return finalize(align1, align2)
    
        if remove_punctuation:
            text1 = self._remove_punctuations(text1)
            text2 = self._remove_punctuations(text2)

        if lowercase:
            text1 = text1.lower()
            text2 = text2.lower()

        if convert_line_breaks:
            text1 = text1.replace("\n", " ")
            text2 = text2.replace("\n", " ")

        aligned_data = needle(
            self._remove_multiple_spaces(text1).split(" "), 
            self._remove_multiple_spaces(text2).split(" ")
            )

        return aligned_data

    def create_op_data(self, asr_text, ref_text, with_timings=True):
        """
        Compares ASR text with some referential text and returns data structure 
        with information on the alignment. The structure includes for every element 
        in the aligned list:
        - operation: 
        - token from text1
        - token from text2
        - token absolute start time
        - token duration
        - pause before token
        - pause after token
        The last four parameters are added if 'with_timings' is True
        Args:
        - asr_data: ASR text
        - ref_data: REF text
        - with_timings: if True, time attributes (start time, duration, pause before, pause after) 
          are added to the resulting data structure
        Results:
        - a list with op data
        - a list with errors 
        """

        def get_value(elm_id, phase):
            return float(self.transcript[elm_id][phase])

        def count_INS(transcript):
            return len([el for el in transcript if el=='-'])

        #asr_text = "Danes je lep sončen dan."
        #ref_text = "Dane jep lep sončen dan"

        res = self.align_text(asr_text.strip(), ref_text.strip())

        offset = 0

        op_data = []
        error_data = []
        deleted_tokens = 0

        for i in range(0, len(res[2])):
            if res[2][i] == res[3][i]:
                op = 'OK'
            elif res[2][i]=='-':
                op='INS'
                offset -= 1
            elif res[3][i]=='-':
                op='DEL'
            else:
                op='SUB'

            if res[2][i] not in ('-', self._remove_punctuations(self.transcript[i+offset]['text'].lower())):
                error_data.append(dict(
                    error='no matching token in the asr timings',
                    ref_token=res[2][i],
                    asr_token=res[3][i],
                    asr_timings_token=self.transcript[i+offset]['text']
                    ))
                
            if with_timings:
                try:
                    w_pos = get_value(i+offset, 'pos')
                    w_start = get_value(i+offset, 'startTime')
                    w_end =  get_value(i+offset, 'endTime')
                    w_len = round(w_end - w_start, 3)
                    
                    if op=='DEL':
                        deleted_tokens+=1
                    
                    if i==0:
                        w_pause_before = 0
                    else:
                        w_pause_before = round(w_start - get_value(i+offset-1, 'endTime'), 3)
                    
                    if i+offset>=len(self.transcript)-1:
                        w_pause_after = 0
                    else:
                        w_pause_after = round(get_value(i+offset+1, 'startTime') - w_end, 3)
                        
                    op_data.append(dict(
                        operation=op,
                        ref_token=res[3][i],
                        asr_token=res[2][i],
                        start_time=w_start,
                        duration=w_len,
                        pause_before=w_pause_before,
                        pause_after=w_pause_after,
                        pos=w_pos,
                        pos_index=i-deleted_tokens
                    ))
                    
                except Exception as e:
                    self.logger.error(f"Error when creating OP file. Exception: {e}")
                    self.logger.error(f"asr_roken={res[2][i]}, ref_token={res[3][i]}, step={i}, offset={offset}, asr_timings_token={self.transcript[i+offset]['text']}")

            else:
                op_data.append(dict(
                        operation=op,
                        ref_token=res[3][i],
                        asr_token=res[2][i],
                    ))

        return op_data, error_data

    def find_spoken_token(self, ct, from_el, to_el):
        """
        Returns token that is spoken at time 'ct'. Bisection is used to make search fast.
        Args:
        - ct: time in miliseconds
        - from_el: id of the element to look from
        - to_el: id of the element to look to
        """
        #print(from_el, to_el)
        mid_el_num = from_el + int((to_el - from_el) / 2)

        if from_el == to_el or from_el==mid_el_num or to_el==mid_el_num:
            return None

        mid_ts_start = self.transcript[mid_el_num]['startTime']
        mid_ts_end = self.transcript[mid_el_num]['endTime']

        if ct < mid_ts_start:
            return self.find_spoken_token(ct, from_el, mid_el_num)
        elif ct <= mid_ts_end:
            return (
                self.transcript[mid_el_num]['text'], 
                self.transcript[mid_el_num]['pos']
                )
        else:
            return self.find_spoken_token(ct, mid_el_num, to_el)

    def find_token_by_pos(self, pos, from_el, to_el):
        """
        Returns text, startTime and endTime of the token that is positioned at position 
        pos in the current editor content
        Args:
        - pos: absolute position in the content
        - from_el: id of the element to look from
        - to_el: id of the element to look to
        """
        mid_el_num = from_el + int((to_el - from_el) / 2)

        if from_el == to_el or from_el==mid_el_num or to_el==mid_el_num:
            return None

        mid_el_start_pos = self.transcript[mid_el_num]['pos']
        mid_el_end_pos = mid_el_start_pos + len(self.transcript[mid_el_num]['text'])

        if pos < mid_el_start_pos:
            return self.find_token_by_pos(pos, from_el, mid_el_num)
        elif pos <= mid_el_end_pos:
            return (
                self.transcript[mid_el_num]['text'], 
                self.transcript[mid_el_num]['startTime'],
                self.transcript[mid_el_num]['endTime'],
                )
        else:
            return self.find_token_by_pos(pos, mid_el_num, to_el)

    def index_tokens(self, plaintext):
        """
        Returns a list with absolute positions of tokens in the plaintext.
        Args:
        - plaintext: text to index
        """
        #return [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        pos = [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        # remove positions for multiple spaces - only last space in the row is relevant
        pos_cleaned = []
        for i in range(0, len(pos)):
            if i<len(pos)-1 and pos[i]==pos[i+1]:
                continue
            else:
                pos_cleaned.append(pos[i])
        return pos_cleaned

    def token_positions(self, plaintext):
        """
        """
        indexed_tokens = dict()
        # get positions of spaces
        pos = [0]+[el.span()[1] for el in re.finditer(" ", plaintext)]
        # remove positions for multiple spaces - only last space in the row is relevant
        pos_cleaned = []
        for i in range(0, len(pos)):
            if i<len(pos)-1 and pos[i]==pos[i+1]:
                continue
            else:
                pos_cleaned.append(pos[i])
        # create dict, keys must be without punctuations
        plaintext_cleaned = self._remove_multiple_spaces(plaintext)
        for c, token in enumerate(plaintext_cleaned.split(" ")):
            indexed_tokens[self._remove_punctuations(token)]=pos_cleaned[c]

        return indexed_tokens

class TruebarSTT():

    # initialize 
    def __init__(
        self, 
        TB_config, 
        logger_folder=os.path.dirname(__file__)
        ):
        self.token_attributes = {}
        self.configuration_attributes = {}
        self.session = None
        self.last_sessionId = None
        self.session_start = None
        self.elapsed_time_to_connect = None # tells how much time was needed last time to connect - this is kind of ping
        self.session_info = []         # keeps info about all sessions performed with the object
        self.connection_errors = []    # keeps info about failed attempts if any during establishing the last session
        self.TB_config = TB_config
        self.sender = None
        self.listener = None
        self.state = None              # added to keep info on session state. Could be one of the following: {None, Authenticated, Connected, Error}
        self.group_users = []
        self.timeout = self.TB_config.pop('timeout', None)
        self.data = queue.Queue()
        self.status = queue.Queue()
        self.microphone_stream = None
        self.input_devices = self.get_input_devices()
        self.selected_input_device_index = None   # if None, default system input will be used
        self.logger = setup_logging("TB_library", logger_folder)

    # get IP address of the client
    def get_ip(self):
        """
        Return IP of the local machine
        """
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.settimeout(0)
        try:
            # doesn't even have to be reachable
            s.connect(('10.254.254.254', 1))
            IP = s.getsockname()[0]
        except Exception:
            IP = '127.0.0.1'
        finally:
            s.close()
        return IP
        
    # check for local Root certificate if exists
    def local_root_certificate(self):
        if os.path.isfile(os.path.join(os.path.dirname(__file__), 'cert', 'ISRGRootX1.pem')):
            return os.path.join(os.path.dirname(__file__), 'cert', 'ISRGRootX1.pem')
        else:
            return True

    # authenticate to get a token
    def get_auth_token(self):
        
        verify = self.local_root_certificate()
        if verify != True:
            self.logger.debug(f"Verification of root certificate via local file at {verify}")
            
        url = self.TB_config.get("auth-url", None)
        headers = CaseInsensitiveDict()
        headers["Content-Type"] = "application/x-www-form-urlencoded"
        headers["Content-Length"] = "0"
        data = "grant_type=password&username="+self.TB_config["username"]+"&password="+self.TB_config["password"]+"&client_id=truebar-client"

        try:
            resp = requests.post(url, headers=headers, data=data, verify=verify)
        except Exception as e:
            self.connection_errors.append(dict(messageType='ERROR', message="Error sending POST request for authentication!", exception=e))
            self.logger.debug("Error sending POST request for authentication! Exception: %s" %e)
            self.state = None
            return None

        if resp.status_code not in (200, 201, 204):
            self.connection_errors.append(dict(messageType='ERROR', status_code=resp.status_code, message=f"Access denied."))
            self.logger.debug(f"Access denied. Response code: {resp.status_code}")
            self.state = None
            return None
        else:
            self.token_attributes = resp.json()
            self.state = 'Authenticated'
            return resp.json()["access_token"]

    # get info about available input devices
    def get_input_devices(self):
        input_devices = dict()
        p = pyaudio.PyAudio()
        info = p.get_host_api_info_by_index(0)
        numdevices = info.get('deviceCount')
        for i in range(0, numdevices):
            if (p.get_device_info_by_host_api_device_index(0, i).get('maxInputChannels')) > 0:
                device_name = p.get_device_info_by_host_api_device_index(0, i).get('name')
                if re.search(r"^microsoft sound mapper", device_name.lower()):
                    continue
                input_devices[i] = p.get_device_info_by_host_api_device_index(0, i).get('name')
                #print("Input Device id ", i, " - ", p.get_device_info_by_host_api_device_index(0, i).get('name'))
        return input_devices

    # read configuration /must be authenticated
    def get_configuration(self):
        if not self.token_attributes.get("access_token", False):
            print("To read configuration, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        #url = "https://"  + self.TB_config["ASR_server"] + "/api/client/editor/configuration"
        url = "https://"  + self.TB_config["ASR_server"] + "/api/client/configuration"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())
        
        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error reading configuration!")
            self.logger.error(resp.status_code)
            self.logger.error(codecs.decode(resp.status_code, 'UTF-8'))
            return False
        else:
            self.configuration_attributes = resp.json()
            return resp.json()

    # DEPRECATED!
    # check available model. Returns a list of pairs (language-code, domain-code) 
    def get_available_models_deprecated(self):
        if not self.token_attributes.get("access_token", False):
            print("To read configuration, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        available_models = []
        config = self.get_configuration()
        for lang in config.get("transcriptionLanguageOptions"):
            for domain in lang.get("domains"):
                if domain["allowed"]: available_models.append((lang["languageCode"], domain["domainCode"]))
        return available_models

    # check available model. Returns a list of tuples (framework, language, domain, model)
    def get_available_models(self):
        stt_status = self.get_stt_status()
        available_models = []

        for f in stt_status["frameworks"]:
            fcode = f["code"]
            for l in f["languages"]:
                lcode = l["code"]
                for d in l["domains"]:
                    dcode = d["code"]
                    for m in d["models"]:
                        mcode = m["code"]
                        available_models.append(dict(
                            framework=fcode,
                            language=lcode,
                            domain=dcode,
                            model=mcode
                        ))
        return available_models

    # get stt status
    def get_stt_status(self):
        if not self.token_attributes.get("access_token", False):
            print("To update configuration, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + "/api/stt/status"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        headers["Content-Type"] = "application/json"

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 204):
            print("Error in setting configuration!")
            print(resp.status_code)
            exit()
        else:
            return resp.json()

    # check model versions for a specific language and domain
    def get_model_versions(self, langcode, domaincode):
        if not self.token_attributes.get("access_token", False):
            print("To read configuration, user must be authenticated! Check method get_auth_token().")
            exit()

        config = self.get_configuration()
        versions = []
        for lang in config.get("transcriptionLanguageOptions"):
            if lang["languageCode"]==langcode:
                for domain in lang.get("domains"):
                    if domain["domainCode"]==domaincode:
                        for ver in domain["models"]:
                            if ver["allowed"] and ver["available"]:
                                versions.append(ver["version"])
        return versions
    
    # update configuration
    def update_configuration(self, new_config):
        if not self.token_attributes.get("access_token", False):
            print("To update configuration, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + "/api/client/configuration"
        #url = "https://demo-api.true-bar.si/api/client/editor/configuration"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        headers["Content-Type"] = "application/json"

        data = new_config

        resp = requests.patch(url, headers=headers, data=data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 204):
            print("Error in setting configuration!")
            print(resp.status_code)
            print(resp.json())
            exit()
        else:
            return True

    # DEPRECATED!
    # public method to set language, domain and model version
    def set_model_deprecated(self, lang, domain, version):
        new_config = str(dict(transcriptionLanguage=lang, transcriptionDomain=domain, transcriptionModelVersion=version)).replace("\'", '"')
        self.update_configuration(new_config)
        return True
    
    def set_model(self, framework, lang, domain, version):
        new_config = dict(
            stt=dict(
                framework=dict(
                    value=framework
                ),
                language=dict(
                    value=lang
                ),
                domain=dict(
                    value=domain
                ),
                model=dict(
                    value=version
                )
            )
        )
        self.update_configuration(json.dumps(new_config))
        return True

    # public method to transcribe from microphone or file
    def transcribe(self, input="microphpne", audiofile=None, outfile=None):
        self.__start_session(input, audiofile, outfile)

    # transcribe from microphone
    def transcribe_from_microphone(self, continue_session=False, keyListener=True):
        return self.__start_session(keyListener=keyListener, continue_session=continue_session)

    # transcribe from a file
    def transcribe_from_file(self, token, file):
        pass

    # private method to stream from microphone
    def __microphone_streamer(self):
        # initiate microphone stream
        audio = pyaudio.PyAudio()
        self.stream = audio.open(format=self.TB_config["audio_format"],
                    channels=self.TB_config["audio_channels"],
                    rate=self.TB_config["audio_rate"],
                    input=True,
                    input_device_index=self.selected_input_device_index,
                    frames_per_buffer=self.TB_config["audio_chunk"])
        # enqueue chunks
        while self.session and self.session.connected:
            data = self.stream.read(self.TB_config["audio_chunk"])
            self.data.put(data)

        # use explicit return, otherwise the thread will not close
        return True

    # private method to enqueue wave file frames
    def __enqueue_wave_frames(self, audiofile):
        # enqueue frames
        with wave.open(audiofile, 'rb') as wf:
            data = wf.readframes(self.TB_config["audio_chunk"])
            while data != b"": #and self.session and self.session.connected:
                self.data.put(data)
                data = wf.readframes(self.TB_config["audio_chunk"])
            self.data.put(b"")
            self.logger.info("file enqueued")

        # use explicit return, otherwise the thread will not close
        return True
                   
    # private method to read data from a queue and send it to the ASR server. The queue is fed by a microphon stream or from a file.
    def __data_sender(self):
        while self.session and self.session.connected:
            try:
                data = self.data.get(block=True, timeout=self.timeout)
                # if reading from file and b"" is encountered, end the session
                if data == b"":
                    self.__end_session()
            except queue.Empty:
                if not self.session or not self.session.connected:
                    response = dict(messageType="ERROR", message="session failure and no chunks in queue")
                    self.status.put(response)
            else:
                if not data or not self.session:
                    break
                try:
                    self.session.send_binary(bytes(data))
                except Exception as e:
                    response = dict(status=4, message=f"session failure, {e}")
                    response = dict(messageType="ERROR", message=f"session failure, {e}")
                    self.status.put(response)
        
        # use explicit return, otherwise the thread will not close
        return True
    
    # public method that listens to responses, received from the ASR
    def __response_listener(self):
        while self.session and self.session.connected:
            try:
                #msg = self.session.recv()
                resp_opcode, msg = self.session.recv_data()
                if resp_opcode == 8 and len(msg) >= 2:
                    close_code = str(struct.unpack("!H", msg[0:2])[0])
                    close_message = msg[2:].decode("utf-8")
                    self.session_info.append(dict(messageType='INFO', close_code=close_code, close_message=close_message))
                    self.logger.debug("websocket closed with code: %s, message: %s" %(close_code, close_message))
                    # session has been closed, passing back to the while loop will stop the listener
                    continue
                
            except websocket.WebSocketTimeoutException:
                self.logger.warning("ws timeout exception occured in listener!")
                pass
            except Exception as e:
                #response = dict(messageType="ERROR", message=f"session failure, {e}")
                #self.status.put(response)
                self.logger.error("Error in listener! Exception: %s" %e)
                break

            # id resp_opcode != 8
            else:
                response = json.loads(msg)
                self.status.put(response)    

        # use explicit return, otherwise the thread will not close
        return True    
    
    # private metod to start a session
    def __start_session(self, input="microphone", audiofile=None, outfile=None, keyListener=False, continue_session=False):
        # STEP 1: open a session and save session info
        #response = {}
        self.logger.info("***************** new session *****************")
        self.logger.debug(f'USER: {self.TB_config["username"]}')
        self.logger.debug(f'IP address: {self.get_ip()}')
        
        # check if proxy
        if self.TB_config.get('proxy', None):
            proxy = self.TB_config['proxy']
            self.logger.info(f"Using proxy: {proxy}")
        else:
            proxy=None

        try:
            self.logger.debug("opening websocket")
            self.session_start = time.time()
            """
            self.session = websocket.create_connection(
                            'wss://' + self.TB_config["ASR_server"] + ':443/ws' +
                            '?access_token=' + self.token_attributes["access_token"]
                        )
            """
            url_new_session = f'wss://{self.TB_config["ASR_server"]}:443/ws?access_token={self.token_attributes["access_token"]}'
            url_continue_session = f'wss://{self.TB_config["ASR_server"]}:443/ws?access_token={self.token_attributes["access_token"]}&session_id={self.last_sessionId}'
             
            if continue_session:
                if not self.last_sessionId:
                    self.logger.warning(f"Client requested to continue last session, but last session unknown!")
                    return False
                elif proxy:
                    self.session = websocket.create_connection(
                        url=url_continue_session, 
                        http_proxy_host=proxy.split(":")[0], 
                        http_proxy_port=proxy.split(":")[-1]
                        )
                else:
                    self.session = websocket.create_connection(
                        url=url_continue_session, 
                        )
            elif proxy:
                self.session = websocket.create_connection(
                        url=url_new_session, 
                        http_proxy_host=proxy.split(":")[0], 
                        http_proxy_port=proxy.split(":")[-1]
                        )
            else:
                self.session = websocket.create_connection(
                    url=url_new_session
                    )

            self.logger.debug("websocket established")
            self.state = 'Connected'
            self.elapsed_time_to_connect = time.time() - self.session_start

        except Exception as e:
            self.connection_errors.append(dict(messageType='ERROR', message="Error establishing websocket connection!", exception=e))
            self.logger.error("Error establishing websocket connection! Exception: %s" %e)
            self.state = 'Error'
            return False

        try:
            resp_opcode, msg = self.session.recv_data()
            # if resp_opcode == 8 --> websocket has been closed
            # see https://websocket-client.readthedocs.io/en/latest/examples.html
            if resp_opcode == 8 and len(msg) >= 2:
                close_code = str(struct.unpack("!H", msg[0:2])[0])
                close_message = msg[2:].decode("utf-8")
                self.connection_errors.append(dict(messageType='INFO', close_code=close_code, close_message=close_message))
                self.logger.info("websocket closed with code: %s, message: %s" %(close_code, close_message))
                return False
            else:
                sinfo=json.loads(msg)
                self.session_info.append(sinfo)
                self.last_sessionId = sinfo.get("sessionId", None)
                if 'messageType' in sinfo and sinfo['messageType']=='ERROR':
                    self.logger.error(f"Handshake OK, but can't start session. Error message: {sinfo.get('message', None)}")
                    self.connection_errors.append(dict(messageType='ERROR', message=sinfo.get('message', None)))
                    return False
                self.logger.debug("session handshake ok")
                self.logger.info("session info: %s" % json.loads(msg))

        except Exception as e:
            self.connection_errors.append(dict(messageType='ERROR', message="Error receiving websocket handshake!", exception=e))
            self.logger.error("Error receiving websocket handshake! Exception %s" %e)
            return False

    
        # STEP 2: start a sender, listener, microphone and keyboard listener in separated threads

        # start a sender
        try:
            self.sender = threading.Thread(target=self.__data_sender)
            self.sender.start()
            self.logger.debug("sender started")
        except Exception as e:
            self.session_info.append(dict(messageType='ERROR', message="Error starting sender in a thread!", exception=e))
            self.logger.error("Error starting sender in a thread! Exception: %s" %e)
            self.state = 'Error'
            return False


        # start a response listener
        try:
            self.listener = threading.Thread(target=self.__response_listener)
            self.listener.start()
            self.logger.debug("listener started")
        except Exception as e:
            self.session_info.append(dict(messageType='ERROR', message="Error starting listener in a thread!", exception=e))
            self.logger.error("Error starting listener in a thread! Exception: %s" %e)
            self.state = 'Error'
            return False

        # start microphone if input is set to "microphone" or start reading from the file
        if input=="microphone":
            if self.selected_input_device_index and self.selected_input_device_index != 0:
                self.logger.debug(f"Starting microphone stream using input device {self.input_devices[self.selected_input_device_index]}")
            else:
                self.logger.debug(f"Starting microphone stream using system default input device")
            try:
                self.sender = threading.Thread(target=self.__microphone_streamer)
                self.sender.start()
                self.logger.debug("microphone stream started")
            except Exception as e:
                self.session_info.append(dict(messageType='ERROR', message="Error starting sender in a thread!", exception=e))
                self.logger.error("Error starting microphone stream in a thread! Exception: %s" %e)
                self.state = 'Error'
                return False
        elif input=="file":
            try:
                self.sender = threading.Thread(target=self.__enqueue_wave_frames, args=[audiofile])
                self.sender.start()
                self.logger.debug("file load to the queue started")
            except Exception as e:
                self.session_info.append(dict(messageType='ERROR', message="Error starting sender in a thread!", exception=e))
                self.logger.error("Error starting microphone stream in a thread! Exception: %s" %e)
                return False
        else:
            self.logger.error("Wrong transcription input! Please use 'microphone' for transcribing microphone stream or 'file' if you want to transcribe an audio file.")
            self.session_info.append(dict(messageType='ERROR', message="Wrong transcription input! Valid values are 'microphone' or 'file'"))
            self.state = 'Error'
            return False


        # listen on keyboard for end session request
        if keyListener:
            listener = keyboard.Listener(on_release=self.__on_keyrelease)
            listener.start()
            self.logger.debug("key listener started")
        
        return True

    # public method to end a session
    def end_session(self):
        self.__end_session()

    # private metod to end a session
    def __end_session(self):
        try:
            self.session.send_binary(b"")
        except Exception as e:
            self.session_info.append(dict(messageType='ERROR', message="Error sending empty chunk to close the session!", exception=e))
            self.logger.error("Error sending empty chunk to close the session! Exception: %s" %e)
            self.state = 'Error'
            return False
            #exit()

        # wait until BE closes the session
        while self.session.connected:
            pass 

        self.logger.debug("session closed")
        self.session_info.append(dict(messageType='INFO', message='Session closed', duration='%0.1f seconds'%(time.time()-self.session_start)))
        self.connection_errors.clear()

        self.session = None
        self.state = 'Authenticated'

    # private method to listen on keyboard for ESC or multimedia key PLAY/PAUSE to stop transcribing
    def __on_keyrelease(self, key):
        if key==Key.media_play_pause:
            self.logger.debug('Multimedia key PLAY/PAUSE detected, stopping listener...')
            # stop listener
            self.__end_session()
            return False
        elif key == keyboard.Key.esc:
            self.logger.debug('\nESC key detected, stopping listener...')
            # stop listener
            self.__end_session()
            return False

    # public method to read transcripts, implemented as a generator
    def get_status(self,block=False):
        try:
            response = self.status.get(block=block,timeout=self.timeout)
        except queue.Empty:
            if not self.listener.is_alive:
                yield dict(status=-1)
            pass
        else:
            yield response

    # public method to upload an audio for transcription
    def upload_audio(self, filename, is_async):
        if not self.token_attributes.get("access_token", False):
            print("To upload audio, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/upload?async={is_async}"
        
        files = {
            'file': ('filename', open(filename, 'rb')),
        }

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.post(url, headers=headers, files=files, verify=self.local_root_certificate())
        
        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error uploading audio!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to get session status after async upload call
    def get_session_status(self, sessionid, full_stack=False):
        if not self.token_attributes.get("access_token", False):
            print("To upload audio, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error reading session status!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            if full_stack:
                return resp.json()
            else:
                return resp.json()['status']

    # public method to read session data once finished
    # TODO: rename to get_session_transcript
    def get_session_data(self, sessionid):
        if not self.token_attributes.get("access_token", False):
            print("To download session transcript, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/transcripts"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error pulling session data!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to get all available metadata on specific session
    # TODO: rename to get_session_data
    def get_session_metadata(self, sessionid):
        if not self.token_attributes.get("access_token", False):
            print("To download session data, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error pulling session metadata!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to get all sessions from the authenticated user
    def get_sessions(
        self,
        name=None,
        labels=[],
        createdAt=None,
        createdBy=None,
        source=None, 
        is_discarded=False,
        status='FINISHED', 
        page=0, 
        size=1, 
        sort='createdAt,desc',
        admin=False,
        ):
        if not self.token_attributes.get("access_token", False):
            print("To read sessions, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        # create filter
        if admin:
            filter = f"is-discarded={is_discarded}"
        else:
            filter = ""
        if name:
            filter += f"&name={name}"
        if createdAt:
            filter += f"&created-at={createdAt}"
        if createdBy:
            filter += f"&created-by-username={createdBy}"
        if source:
            filter += f"&source={source}"
        if status:
            filter += f"&status={status}"
        filter += f"&page={page}"
        filter += f"&size={size}"
        filter += f"&sort={sort}"

        for l in labels:
            filter += f"&label={l}"

        if filter[0]=='&':
            filter = filter[1:]

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions?{filter}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error pulling sessions!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to get client labels
    def get_client_labels(self):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers)

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving client labels!")
            self.logger.error(f"Status code: {resp.status_code}")
            self.logger.error(f"Error message: {resp.json().get('message', '')}")
            return False
        else:
            return resp.json()

    # public method to add new label for the client
    def add_client_label(self, label_code, label_color, isDefault=False):
       
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()
        
        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        json_data = {
            "code": label_code,
            "color": label_color,
            "isDefault": isDefault
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error adding client label: label_code={label_code}, label_color={label_color}, isDefault={isDefault}!")
            self.logger.error(f"Status code: {resp.status_code}")
            self.logger.error(f"Error message: {resp.json().get('message', '')}")
            return False
        else:
            return resp.json()

    # public method to retrieve session labels
    def get_session_labels(self, sessionid):
        #sandbox-api.true-bar.si/api/client/sessions/34/labels
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving session labels!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to add session labels
    def add_session_label(self, sessionid, labelid, is_enabled):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()
        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'labelId': labelid,
            'isEnabled': is_enabled,
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error updating session labels!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # public method to get user replacements
    def get_user_replacements(self):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/replacements"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving session labels!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to post user replacements
    def post_user_replacement(self, replacement):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/replacements"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        json_data = replacement

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error posting replacement!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # public method to delete user replacement
    def delete_user_replacement(self, rid):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/replacements/{rid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.delete(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error deleting replacement!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # public method to patch session name
    def patch_session_name(self, sessionid, name, notes=None):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()
    	
        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        if notes:
            json_data = {
                'name': name,
                'notes': notes,
            }
        else:
            json_data = {
                'name': name
            }

        resp = requests.patch(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error updating session name!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True
    
    # public method to patch session labels
    def patch_session_label(self, sessionid, labelid, is_enabled):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()
    	#http://sandbox-api.true-bar.si/api/client/sessions/833/labels/2
        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/labels/{labelid}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'labelId': labelid,
            'isEnabled': is_enabled,
        }

        resp = requests.patch(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error updating session labels!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # get users in group
    def get_users_in_my_group(self):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/users"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving session labels!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # get session transcript
    def get_session_transcript(self, sessionid, return_type='transcript'):
        """
        Returns transcript in TB format
        If you want to retrieve text only, use return_type=text
        To return list of finals, use return_type=finals
        TODO: this seems to be a duplicate of get_session_data!
        """
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/transcripts"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving session transcript!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            if return_type=='transcript' or return_type not in ('text', 'finals'):
                return resp.json()
            elif return_type=='text':
                transcript=""
                for ide in resp.json():
                    transcript_with_metadata = json.loads(ide["content"])
                    transcript += "".join([el["text"] if not el.get("spaceBefore", False) else " "+el["text"] for el in transcript_with_metadata])                
                return transcript
            elif return_type=='finals':
                finals = []
                for ide in resp.json():
                    transcript_with_metadata = json.loads(ide["content"])
                    finals.append("".join([el["text"] if not el.get("spaceBefore", False) else " "+el["text"] for el in transcript_with_metadata]))
                return finals
            else:
                self.logger.error(f"Unknown value for parameter 'return_type': {return_type}. Function 'get_session_transcript'")
                return False

    # public method to add session share to a user
    def add_session_shares(self, sessionid, userid):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()
    	
        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/shares"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = {
            'userId': userid,
        }

        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error updating session shares!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # public method to retreive session audio
    def get_session_audio(self, sessionid):
        if not self.token_attributes.get("access_token", False):
            print("To download audio, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/sessions/{sessionid}/audio.wav"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error pulling session audio file!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.content

    # public method to play audio
    def play_audio(self, audiofile):
        # open wave file
        wave_file = wave.open(audiofile, 'rb')
        
        # initialize audio
        py_audio = pyaudio.PyAudio()
        stream = py_audio.open(format=py_audio.get_format_from_width(wave_file.getsampwidth()),
                            channels=wave_file.getnchannels(),
                            rate=wave_file.getframerate(),
                            output=True)

        # skip unwanted frames
        #n_frames = int(start * wave_file.getframerate())
        #wave_file.setpos(n_frames)
        n_frames = wave_file.getnframes()

        # write desired frames to audio buffer
        #n_frames = int(length * wave_file.getframerate())
        frames = wave_file.readframes(n_frames)
        stream.write(frames)

        # close and terminate everything properly
        stream.close()
        py_audio.terminate()
        wave_file.close()

    # print error stack
    def print_errorstack(self):
        for el in self.connection_errors:
            print(f"{el}\n") 

    # upload logs
    def upload_logs(self, log_folder, username='marko'):
        """
        This is a custom procedure to upload log files and settings.yaml from a certain directory to the server.
        Files are first copied and renamed to fit destination format, i.e. <name>_<timestamp>_<filename>.{log|yaml}
        Notice that temp files are deleted at beggining if there are any from the previous upload.
        """
        if not self.token_attributes.get("access_token", False):
            print("To upload files, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://logs-uploader.true-bar.si/upload"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        counter = 0
        failed = 0

        # remove previous temp files if exist in the log folder
        for file in os.listdir(log_folder):
            if file[0]=="_":
                os.remove(os.path.join(log_folder, file))
        
        for file in os.listdir(log_folder):
            if file[-4:].lower() in ('.log', 'yaml'):
                counter += 1
                # make a copy of the log file and rename it to fir the destination format
                current_time_string = time.strftime("%Y%m%d_%H%M%S", time.localtime())
                destination_filename = os.path.join(log_folder, f"_{username}_{current_time_string}_{file}")

                shutil.copyfile(
                    os.path.join(log_folder, file), 
                    destination_filename
                    )
                files = {'file': open(destination_filename, 'rb')}

                resp = requests.post(
                    url=url, 
                    headers=headers, 
                    files=files,
                    verify=False
                    )

                if resp.status_code not in (200, 201, 204):
                    failed += 1
                    self.logger.error(f"Error uploading log file {file}")
                    self.logger.error(resp.status_code)
                    if resp.json(): self.logger.error(resp.json().get("message", ""))
                else:
                    self.logger.debug(f"Log file {file} uploaded.")

        return dict(all_log_files=counter, failed=failed)

    ### working with dictionary
    # public method to get phoneset
    def get_model_phoneset(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To download audio, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/phone-set"

        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving phone-set for the model!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # public method to read frequency classes for the dictionary
    def get_frequency_classes(self):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/frequencyClasses"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error("Error retrieving frequency classes!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # get all data for specific word
    def read_dictionary(self, language, domain, model, token):
        #https://demo-api.true-bar.si/api/client/dictionary/languages/sl-SI/domains/MED/dmodels/20221201-1735/words?search-by=word&text=ohranjena 
        #[{"id":10477891,"frequencyClassId":3,"status":"IN_DICTIONARY","text":"ohranjena","pronunciations":[{"saved":true,"text":"ohranjena"}]}]
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words?search-by=word&text={token}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error retrieving dictionary data for token {token}!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # save word entry to dictionary
    def save_dictionary(self, language, domain, model, word_entry):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]

        json_data = word_entry
        
        resp = requests.post(url, headers=headers, json=json_data, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error saving dictionary entry {word_entry}!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # get model info
    def get_model_info(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error reading model statistics!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()
    
    # get model statistics
    def get_model_stat(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/stats"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error reading model statistics!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # get model words count
    def get_model_word_count(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/words/count"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error reading model statistics!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()

    # start model update
    def start_model_update(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/regeneration/regenerate"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.post(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error starting model update!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return True

    # check model update status
    def check_model_update_status(self, language, domain, model):
        if not self.token_attributes.get("access_token", False):
            print("To use this method, user must be authenticated! Check the TruebarSTT method get_auth_token().")
            exit()

        url = "https://"  + self.TB_config["ASR_server"] + f"/api/client/dictionary/languages/{language}/domains/{domain}/dmodels/{model}/regeneration/status"
        
        headers = CaseInsensitiveDict()
        headers["Authorization"] = "Bearer "+self.token_attributes["access_token"]
        
        resp = requests.get(url, headers=headers, verify=self.local_root_certificate())

        if resp.status_code not in (200, 201, 204):
            self.logger.error(f"Error checking model update status!")
            self.logger.error(resp.status_code)
            if resp.json(): self.logger.error(resp.json().get("message", ""))
            return False
        else:
            return resp.json()