"""
This is a simple program that sends audio stream captured over a microfon, to the TB ASR service.
When finals appear, they are pasted in window that currently has focus (at the position of the cursor).

GUI for mainwindow (toolbar) and command helper is designed with QT Designer and generated using "pyuic5"
Generated gui classes are inherited so that whatever change is made, it is not on the generated code.
Due to the problem of DPI unawareness, GUI automatic sizing is achieved with the scale function that
requires each absolute size or position to be first scaled. To this end, do the following:
1. In all modules with generated GUI classes import scaling function: 
    from gui.gui_scaling import scale
2. With regex based find and replace make changes on the generated classes:
    - regex 1: (move|resize|QRect|setContentsMargins)(?!scale)([(\s])(\d+) --> $1$2scale($3)
    - regex 2: (scale\(\d+\),)(\s)(\d+) --> $1$2scale($3) 
3. All icons and images must have relative paths. Use this - with the IgnoreCase switch:
    - regex 1: (QtGui.QPixmap\()("[\da-z_-]+\.)(png|jpg|jpeg|ico)(")(\)) --> QtGui.QPixmap(os.path.join(basedir, $2$3"))

Resources:
- discussion on how to get active window: https://stackoverflow.com/questions/10266281/obtain-active-window-using-python


Author: Vitasis, Inc
Date: October 2022

TODO:
[ ] Some of initializations that are currently under Ui_MainWindow, do no belong there. Consider
    to move under __main__
[ ] Instead of using find and replace to include scaling function, create a python script that will do the same and
    will be then used immediately after gui code is generated.
"""

from asyncio.log import logger
#from curses.ascii import isdigit
from email.mime import base
from math import hypot
from msilib.schema import Dialog
from xmlrpc.client import Boolean
from PyQt5 import QtCore, QtGui, QtWidgets
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
from PyQt5.QtCore import *

import traceback
import sys
import re
from sys import platform
import pyperclip
import keyboard
import json
import time
import os
import yaml
import logging
import win32gui
import webbrowser
from ctypes import windll
import shutil
import threading
import copy
import uuid

from utils import truebar

# win32com.client is required for MSMQ
import win32com.client
import winsound
import pythoncom
from gui.gui_dialogs import Ui_Dialog_commands, Ui_Dialog_sendLog, Ui_Dialog_detailed_settings, Ui_Dialog_add_word, show_message_box
from gui.gui_toolbar import Ui_Toolbar
from gui.gui_scaling import scale, scalingFactor
from gui.notification import create_pixmap, NotificationButton

basedir = os.path.dirname(__file__)
user_appdata = os.path.join(os.getenv('APPDATA'), 'vitasis', 'tb-listener')

# Set to true if you want external commands to be desplayed on the Toolbar
_integration_debug_mode = False

app_version = '23.08'
"""
Version info
publish date: 11. 5. 2023, 
- Buffer status and content reset  (issue #20)
"""

_main_window_height_L = 770
_main_window_height_S = 68
_main_window_width_L = 426
_main_window_width_S = 284
_interim_display_len = 37
_dictionary_frame_height_L = 512
_dictionary_frame_height_S = 439

# colors
cl_light_orange_string = 'rgb(254, 241, 230)'
cl_light_green_string = 'rgb(234, 241, 221)'
cl_light_yellow_string = 'rgb(255, 242, 204)'
cl_message_failed = 'rgb(254, 0, 0)'
cl_message_success = 'rgb(112, 173, 72)'

# word_numbers dict
word_numbers = dict(
    nič=0, nula=0, ena=1, dva=2, tri=3, štiri=4, pet=5, šest=6, sedem=7, osem=8, devet=9, deset=10, 
    enajst=11, dvanajst=12, trinajst=13, štirinajst=14, petnajst=15, šestnajst=16, sedemnajst=17, osemnajst=18, devetnajst=19, dvajset=20,
    enaindvajset=21, dvaindvajset=22, trindvajset=23, štiriindvajset=24, petindvajset=25, šestindvajset=26, sedemindvajset=27, osemindvajset=28, devetindvajset=29, trideset=30, 
    enaintrideset=31, dvaintrideset=33, trintrideset=33, štiriintrideset=34, petintrideset=35, šestintrideset=36, sedemintrideset=37, osemintrideset=38, devetintrideset=39, štirideset=40,
    enainštirideset=41, dvainštirideset=42, trinštirideset=43, štiriinštirideset=44, petinštirideset=45, šestinštirideset=46, sedeminštirideset=47, oseminštirideset=48, devetinštirideset=49, petdeset=50,
    enainpetdeset=51, dvainpetdeset=52, trinpetdeset=53, štiriinpetdeset=54, petinpetdeset=55, šestinpetdeset=56, sedeminpetdeset=57, oseminpetdeset=58, devetinpetdeset=59, šestdeset=60,
    enainšestdeset=61, dvainšestdeset=62, trinšestdeset=63, štiriinšestdeset=64, petinšestdeset=65, šestinšestdeset=66, sedeminšestdeset=67, oseminšestdeset=68, devetinšestdeset=69, sedemdeset=60,
    enainsedemdeset=71, dvainsedemdeset=72, trinsedemdeset=73, štiriinsedemdeset=74, petinsedemdeset=75, šestinsedemdeset=76, sedeminsedemdeset=77, oseminsedemdeset=78, devetinsedemdeset=79, osemdeset=60,
    enainosemdeset=81, dvainosemdeset=82, trinosemdeset=83, štiriinosemdeset=84, petinosemdeset=85, šestinosemdeset=86, sedeminosemdeset=87, oseminosemdeset=88, devetinosemdeset=89, devetdeset=60,
    enaindevetdeset=91, dvaindevetdeset=92, trindevetdeset=93, štiriindevetdeset=94, petindevetdeset=95, šestindevetdeset=96, sedemindevetdeset=97, osemindevetdeset=98, devetindevetdeset=99, sto=100,
)

# logger
def setup_logging(logger_name, logger_folder):
    # read logger config and make sure to use correct logger folder
    with open('./conf/config-logger.yaml', 'r', encoding="utf-8") as stream:
        cfg = yaml.load(stream, Loader=yaml.FullLoader)
    cfg['handlers']['info_file_handler']['filename'] = os.path.join(logger_folder, 'info.log')
    cfg['handlers']['debug_file_handler']['filename'] = os.path.join(logger_folder, 'debug.log')
    cfg['handlers']['error_file_handler']['filename'] = os.path.join(logger_folder, 'error.log')
    logging.config.dictConfig(cfg)
    return logging.getLogger(logger_name)

# settings
def read_config(config_file=os.path.join(user_appdata, "settings.yaml")):
    # if settings file is not found on the user AppData location, read from original version from install folder
    if not os.path.isfile(os.path.join(user_appdata, "settings.yaml")):
        if os.path.isfile(os.path.join(basedir, "settings.yaml")):
            config_file=os.path.join(basedir, "settings.yaml")
        else:
            logger.warning(f"Cannot read settings.yaml file! File {config_file} does not exist!")
            return None

    logger.debug(f"Reading settings from {config_file}")
    with open(config_file, "r", encoding="utf-8") as stream:
        try:
            return yaml.safe_load(stream) 
        except yaml.YAMLError as exc:
            logger.warning(f"Cannot read settings.yaml file, {exc}")
            return None

def save_config(cdict):
    with open(os.path.join(user_appdata, "settings.yaml"), "w", encoding="utf-8") as stream:
        try:
            yaml.safe_dump(cdict, stream, default_flow_style=False, sort_keys=False)
            logger.debug(f"Settings successfully saved to {user_appdata}.")
            return True 
        except yaml.YAMLError as exc:
            logger.warning(f"Cannot update settings.yaml file, {exc}")
            return None

def get_settings_value(dict, key_composition=[]):
    """
    key_composition is a key sequence that coresponds to the key hierarch in the yaml file
    Example: ['general', 'do-postprocessing', 'spp-codes'] refers to the key 'spp-codes' under 'do-postprocessing'...

    """
    if len(key_composition)==0 or key_composition[0] not in dict:
        return False
    elif len(key_composition)==1:
        return dict[key_composition[0]]
    else:
        value = dict[key_composition[0]]
        key_composition.pop(0)
        return get_settings_value(value, key_composition)
        
def set_focus_to_window(window_title_ptr):
    def process_window(hwnd, window_title_ptr):
        res = re.match(window_title_ptr, str(win32gui.GetWindowText(hwnd)))
        if res is not None:
            #print(dict(handle=hwnd, title=str(win32gui.GetWindowText(hwnd))))
            matching_windows.append(dict(handle=hwnd, title=str(win32gui.GetWindowText(hwnd))))
            #win32gui.SetForegroundWindow(hwnd) 
            #return
            #        
    matching_windows = []
    win32gui.EnumWindows(process_window, window_title_ptr)

    if len(matching_windows)==0:
        logger.debug(f"Cannot focus to {window_title_ptr}. Window with such title does not exist!")
    elif len(matching_windows)==1:
        try:
            #win32gui.SetForegroundWindow(matching_windows[0]['handle'])
            set_focus_to_window_handle(matching_windows[0]['handle'])
        except Exception as e:
            logger.warning(f"Can't focus to window with pattern {window_title_ptr}. Exception: {e}")
    else:
        logger.debug(f"Multiple windows found with title: {window_title_ptr}. Focusing to first one!")
        try:
            #win32gui.SetForegroundWindow(matching_windows[0]['handle'])
            set_focus_to_window_handle(matching_windows[0]['handle'])
        except Exception as e:
            logger.warning(f"Can't focus to window with pattern {window_title_ptr}. Exception: {e}")

def set_focus_to_window_handle(handle):
    # This two lines are required otherwise you might not be able to set the focus to the destination window
    # Instead of SendKeys you can also try to use SendWait, which will do the same but also wait for the keystroke messages to be processed
    # see also here:
    # - https://learn.microsoft.com/en-us/dotnet/api/system.windows.forms.sendkeys.sendwait?view=windowsdesktop-7.0
    shell = win32com.client.Dispatch("WScript.Shell")
    shell.SendKeys('')
    try:
        #win32gui.SetForegroundWindow(handle)
        windll.user32.ShowWindow(handle, 9)
        windll.user32.SetForegroundWindow(handle)
    except Exception as e:
        logger.warning(f"Can't focus to window with handle {handle}. Exception: {e}")

def _nvl(t, rv):
    if t:
        return t
    else:
        return rv

def get_model_update_status(language, domain, model, tb_object):
    model_updater_status = tb_object.check_model_update_status(language=language, domain=domain, model=model)
    if not model_updater_status:
        return 'UNKNOWN'
    else:
        return model_updater_status.get('statusCode', 'UNKNOWN')

# classes
class WorkerSignals(QObject):
    """
    Defines the signals available from a running worker thread.
    Supported signals are:
    - finished: No data
    - error: tuple (exctype, value, traceback.format_exc() )
    - result: object data returned from processing, anything
    - status: a dict telling the status of the worker progress

    Check here to see how to send signals in both directions, i.e. 
    from the main GUI to worker and back, from the worker to GUI
    https://www.pythonguis.com/faq/pause-a-running-worker-thread/
    """
    finished = pyqtSignal()
    error = pyqtSignal(tuple)
    result = pyqtSignal(object)
    status = pyqtSignal(dict)

class Streamer(QRunnable):
    def __init__(
        self, 
        stt, 
        settings=None,
        spp_codes=None,
        dictation_type="rtf",
        capslock='off', 
        shift='on',
        continue_session=False,
        buffering=None,
        buffer="",
        dictaphone_mode=False,
        templates=None,
        ):
        super(Streamer, self).__init__()
        self.signals = WorkerSignals()
        self.stt = stt
        self.templates = templates
        self.commands_helper_active = False
        self.dictaphone_mode = dictaphone_mode
        # set platform and copy/paste key combination
        # TODO: should go to config
        if platform == "linux" or platform == "linux2":
            self.platform = 'Linux'
            self.copypaste = 'cmd+v'
            self.undo = 'cmd+z'
        elif platform == "darwin":
            self.platform = 'Mac OS'
            self.copypaste = 'cmd+v'
            self.undo = 'cmd+z'
        elif platform == "win32":
            self.platform = "Windows"
            self.copypaste = 'ctrl+v'
            self.undo = 'ctrl+z'
        # other settings
        self.settings = settings
        self.translations = self.create_translations_dict()
        self.spp_codes = spp_codes
        self.dictation_type = dictation_type
        self.caps_lock_state = capslock
        self.shift_state = shift
        self.ins_state = 'off'
        self.start_with_space = False
        self.spelling = False
        self.command_stack = []
        self.tb_platform = "sandbox"
        self.last_final = None
        self.last_interim = None
        self.command_composite = None
        self.continue_session = continue_session
        self.buffer = buffer
        self.buffering = buffering
        self._ptr_command = re.compile(r'(\s)*(<[^>\s]+>)(\s)*')
        self._ptr_command_composition = re.compile(r'^(<[^>\s]+>)(\s)*([\d]+[\.\,\?\!]*|[a-zščž]+[\.\,\?\!]*)$', flags=re.IGNORECASE)
        self._ptr_nl = re.compile(r'(\s)*(<nl>)(\s)*')
        self._ptr_np = re.compile(r'(\s)*(<np>)(\s)*')
        self._ptr_numbers_to_join = re.compile(r'(^|\s)(\d+)(\s)(\d+)')
        self._ptr_numbers_to_translate1 = re.compile(r'(\d+)(\s)(tisoč|sto|deset)(\s|\,\.\!\?)', flags=re.IGNORECASE)
        self._ptr_numbers_to_translate2 = re.compile(r'(tisoč|sto|deset|nič|nula|dva|tri|štiri|pet|šest|sedem|osem|devet)(\s)(\d+)', flags=re.IGNORECASE)
        self._ptr_leading_zero = re.compile(r'(^|\s)(nič)(\s)(\d+)', flags=re.IGNORECASE)
        self._ptr_spp_code = re.compile(r'^(\s)*([a-z])(\s)*(\d{1,2})(\.\s\d{1,2})*(\s|$)', flags=re.IGNORECASE)
        self._ptr_nl_as_word = re.compile(r'(.*)(\snova\svrsta\s|\snova\svrstica\s)(.*)', flags=re.IGNORECASE)
        self._ptr_np_as_word = re.compile(r'(.*)(\snov\sodstavek\s|\snovi\sodstavek\s)(.*)', flags=re.IGNORECASE)

    def __nvl(self, t):
            if t:
                return t
            else:
                return ""

    def __word_numbers(self, t):
        """
        This function is used to clean token of punctuations and return number as digit if the token is numeric
        Useful for commands with arguments that are numbers, such as <ins> 5 
        """
        t_clean = re.sub(r"([^\.\,\!\?]+)([\.\,\!\?]+)", r"\g<1>", t).lower()
        return(str(word_numbers.get(t_clean, t)))

    def extend_spp_codes(self, final, spp_signal=False):

        res = re.search(self._ptr_spp_code, final)
        # I expect that the code if encountered is represented with the whole final.
        # Given that sometimes it might happen that first group will have more than
        # a single letter, e.g. 'TT 18.7' instead 'T 18.7', I allow for some discepancy 
        # in the length.  

        if res:
            #g1=res.group(2)
            #if res.group(2).lower()=='ipsilon':
            #    g1='Y'
            formated_spp_code = self.__nvl(res.group(2)) + self.__nvl(res.group(4)) + self.__nvl(res.group(5))
            formated_spp_code = formated_spp_code.replace(" ", "").upper()
            logger.debug(f"spp code format detected: {formated_spp_code}")
            if formated_spp_code in self.spp_codes:
                formated_spp_code = f"{formated_spp_code}: {self.spp_codes[formated_spp_code]}"
                logger.debug(f"spp code found: {formated_spp_code}")
                return formated_spp_code
            else:
                logger.debug(f"spp code {formated_spp_code} not in database!")
                
        return final

    def perform_replacements(self, final):
        
        def convert_number(t):
            return word_numbers.get(t.lower(), t)

        # REPLACEMENT 1:
        #   - if two numbers where one written with letters other with digits, like 3 tisoč --> 3000
        old_final = final
        res = re.search(self._ptr_numbers_to_translate2, final)
        while res:
            logger.debug("Applying raplacement rule R1-1 ...")
            num = f"{convert_number(res.group(1))} {res.group(3)}"
            if num==res:
                logger.debug(f"R1 rule failed and exited")
                break
            final = re.sub(self._ptr_numbers_to_translate2, num, final) 
            res = re.search(self._ptr_numbers_to_translate2, final)
        if old_final != final:
            logger.debug(f"R1: {old_final} --> {final}")

        old_final = final
        res = re.search(self._ptr_numbers_to_translate1, final)
        while res:
            logger.debug("Applying raplacement rule R1-2 ...")
            num = f"{res.group(1)} {convert_number(res.group(3))}"
            if num==res:
                logger.debug(f"R2 rule failed and exited")
                break
            final = re.sub(self._ptr_numbers_to_translate1, num, final) 
            res = re.search(self._ptr_numbers_to_translate1, final)
            print(num, final, res)
        if old_final != final:
            logger.debug(f"R1: {old_final} --> {final}")

        # REPLACEMENT 2:
        #   - if leading zero with letters, like 'nič 1' or 'nič 2'
        old_final = final
        res = re.search(self._ptr_leading_zero, final)
        while res:
            logger.debug("Applying raplacement rule R2 ...")
            final = re.sub(self._ptr_leading_zero, f"0{res.group(4)}", final) 
            res = re.search(self._ptr_leading_zero, final)
        if old_final != final:
            logger.debug(f"R2: {old_final} --> {final}")

        # REPLACEMENT 3: 
        #   - if one number after another without a punctuation, like: 20 15 --> change by joining to 2015
        #   - if first number is hundred or thousand, like: 100 15 --> change by sum to 115
        # TODO: if first number is 2, 3, 4, 5, 6, 7, 8, 9 and second is 100 or 1000 --> change by multiplication, e.g. 3 1000 --> 3000
        old_final = final
        res = re.search(self._ptr_numbers_to_join, final)
        while res:
            logger.debug("Applying raplacement rule R3 ...")
            x1 = int(res.group(2))
            x2 = int(res.group(4))
            if x1!=0 and (x1 % 1000 == 0 or x1 % 100 ==0) and len(str(x2))<len(str(x1)):
                num = self.__nvl(res.group(1)) + str(x1+x2)
            else:
                num = self.__nvl(res.group(1)) + res.group(2)+res.group(4)
            final = final[0:res.span()[0]] + num + final[res.span()[1]:]
            res = re.search(self._ptr_numbers_to_join, final)

        if old_final != final:
            logger.debug(f"R3: {old_final} --> {final}")

        # USER DEFINED REPLACEMENTS
        logger.debug("Applying user-defined raplacements ...")
        for rpl in self.settings["replacements"]:
            old_final = final
            if rpl["ignorecase"]:
                ptr=re.compile(rf"{rpl['source']}", re.IGNORECASE)
            else:
                ptr=re.compile(rf"{rpl['source']}")
            final = re.sub(ptr, rpl["target"], final)
            if old_final != final:
                logger.debug(f"Custom replacement {rpl}: {old_final} --> {final}") 
        return final

    def format_notes(self, final):
        if final.strip().lower()=='opomba':
            return "===================== OPOMBA ====================="
        elif final.strip().lower()=='konec opombe':
            return "=================================================="
        else:
            return final 

    def get_text(self, response):
        """
        get_text only process finals. It does the following:
        - extracts type of response (isFinal: True/False) and associated transcript, if any. This is done by the subproces formated_final()
        If isFinal:
        - checks if transcript is a space: in this case makes a space (pyperclip + keyboard) and returns True
        - checks if transcript is not an empty string. In this case it logs the final saves the final in the self.last_final
        - checks helper_window is active - in this case it ignores everything except teh command 'close' 
        - checks one of the reserved phrases were dictated. In this case, the phrase is processed and get_text returns True
        - checks spelling command was issued. Set spelling mode. get_text returns True
        - checks legal command was detected. In this case, get_text returns True after the command is processed
        - if nothing of the above, it proclaims the transcript as aregular final. In this case processes the final and returns True! 
        TODO:
        [ ] command UC is not appropriately handled. If translated from text, e.g. 'z veliko', it ramains unhandled.
            If caught as <UC>, it gets coverted to <uc> and handled as if 'velike črke' command was given! Needs separate handler!
        """
        def format_final(r):
            """
            The format_final() returns the transcript that is extracted from the service response.
            The return value is a dictionary with two keys:
            - isFinal: tells if the transcript is final
            - transcript: the text
            
            Two additional cases are considered:
            a) the transcript is final and is from the list of allowable commands
            b) the transcript is final and include a command that is either unrecognised or not allowed  
            
            In the first case, the command is lowercased, put on the stack and returned. In the second 
            case, the command is removed and ignored (not put on the stack)
            """
            formated_final = ""
            if r and "transcript" in r: 
                for el in json.loads(r.get("transcript").get("content")):
                    if el.get("spaceBefore"):
                        formated_final = formated_final + " " + el.get("text")
                    else:
                        formated_final = formated_final + el.get("text")

                if not r["isFinal"]:
                    self.last_interim = formated_final
                    return dict(
                        isFinal=False,
                        transcript=formated_final
                    ) 
                else:
                    logger.debug("----------------- new final -----------------")
                    logger.debug(f"ASR response: {formated_final}")

            # this is specific case: ASR returns one space if 'presledek' token was used
            if formated_final == " ":
                self.command_stack.append(formated_final)
                return dict(
                        isFinal=True,
                        transcript=" "
                    )

            # if command, do nothing further and return formated command
            if formated_final.lower() in self.settings["legal-commands"][self.dictation_type]:
                self.command_stack.append(formated_final.lower())
                if formated_final.lower() in ('<nl>', '<np>'):
                    self.start_with_space=False
                return dict(
                    isFinal=True,
                    transcript=formated_final.lower()
                )

            else:
                self.command_stack.append(formated_final)

            # Commands compositions, such as <del> 5, <ins> 2 ...  
            cmd = re.search(self._ptr_command_composition, formated_final)
            if cmd and cmd.group(1) in self.settings["legal-commands"][self.dictation_type] and cmd.group(1) in self.settings["composite-commands"]:
                command_argument=self.__word_numbers(formated_final.split(">")[1].strip(" .,!?"))
                print(command_argument)
                if command_argument.isnumeric():
                    logger.debug(f"Command composition detected: {formated_final}")
                    self.command_stack.append(formated_final.lower())
                    return dict(
                        isFinal=True,
                        command=formated_final.split(">")[0].strip()+">",
                        command_argument=command_argument,
                        transcript=formated_final.lower()
                    )    

            # Check for possible tokens "nova vrsta", "nova vrstica", "nov odstavek"
            # TODO: this should better be a parameter in settings so that you can switch off this translations
            #       IMPORTANT: it seems that such cases could be handled efficiently with REPLACEMENTS. Tested
            #       on SANDBOX and works rather fine.
            #resnl = re.search(self._ptr_nl_as_word, formated_final)
            #if resnl:
            #    oldff = formated_final
            #    formated_final = re.sub(self._ptr_nl_as_word, r'\g<1> <nl> \g<3>', formated_final)
            #    logger.debug(f"New line command detected in text and transformed to <nl>: {oldff} --> {formated_final}")

            # If command detected within final, except for command compositions and <nl>, <np> commands, tokens enclosed in <> are removed
            # Notice that <nl> nad <np> are left if cought in the final since the are handled letter with transformation into \n and \n\n
            # TODO: there are other cases when legal command got cought within text and needs to be handled and not just removed. 
            rescmd = re.search(self._ptr_command, formated_final) 
            if rescmd and rescmd.group(2).strip().lower() not in ('<np>', '<nl>'):
                logger.debug(f"Unsupported command detected: {formated_final}")
                formated_final = re.sub(self._ptr_command, '\g<1>\g<3>', formated_final)
            
            if formated_final.strip() == "":
                return dict(
                    isFinal=True,
                    transcript=""
                )

            # if sentence, check space before and return
            if self.start_with_space:
                return dict(
                        isFinal=True,
                        transcript=formated_final
                )
            else:
                # apart from few exceptions that are related to the use of regular tokens as commands, 
                # regular sentences should start with an empty character/space
                if formated_final.lower().strip() != self.settings["user-defined-commands"]["start-buffering"]:
                    self.start_with_space=True
                return dict(
                        isFinal=True,
                        transcript=formated_final.lstrip()
                )
                
        final = ""

        if response and "transcript" in response: 

            hypothesis = format_final(response)

            # if the hypothesis is final    
            # if SPACE, process it immediately
            if hypothesis["isFinal"] and hypothesis["transcript"]==" ":
                if self.buffering:
                    self.buffer += " "
                elif not self.dictaphone_mode:
                    pyperclip.copy(" ")
                    keyboard.press_and_release(self.copypaste)
                return True
            
            if hypothesis["isFinal"] and len(hypothesis["transcript"].strip()) > 0:
                
                final = hypothesis["transcript"]

                # if translations are enabled
                if get_settings_value(self.settings, ['user-defined-commands', 'enable-translations']):
                    final_translated = self.get_translation(final)
                    if final_translated and final_translated != final:
                        logger.debug(f"Translation: {final} --> {final_translated}")
                        final = final_translated

                logger.debug(f"Formatted final: {final}")
                self.last_final=final

                # if commands_helper is active, ignore all finals except the closing command
                if self.commands_helper_active:
                    logger.debug(f"Command helper is active - ignoring all finals.")
                    return True
                
                # check if any of the reserved phrases (notice that some of these reserved phrases are handled on the backend as commands,
                # and could be moved to the code segment, where other commands are manipulated. The reason for these phrases to still be
                # here is since from the start there were no special commands available for them.)
                # 1. START BUFFERING MODE - if not in dictaphone mode
                if not self.dictaphone_mode and final.strip().lower() == self.settings["user-defined-commands"]["start-buffering"]:
                    logger.debug("BUFFERING STARTED ...")
                    self.buffering = True
                    self.buffer = ""
                    print(f"start_with_space = {self.start_with_space}")
                    return True    

                # 2. PAUSE
                if final.strip().lower() == self.settings["user-defined-commands"]["pause-dictation"]:
                    logger.debug("::pause phrase detected")
                    self.stt.end_session()
                    return True
                
                # 3. END DICTATION
                if final.strip().lower() == self.settings["user-defined-commands"]["end-dictation"]:
                    logger.debug("::ending phrase detected")
                    self.stt.end_session()
                    return True

                # 4. EXIT SPELLING MODE
                if self.spelling and final.strip().lower().split(" ")[0] == self.settings["user-defined-commands"]["stop-spelling"]:
                    logger.debug("Stoping dictation...")
                    self.spelling = False
                    return True

                # check and apply allawable commands    
                if final in self.settings["legal-commands"][self.dictation_type]:
                    logger.debug("::command detected")

                    # if in buffering mode, only some commands are eligible
                    if self.buffering and final not in self.settings["legal-commands-when-buffering"]:
                        logger.debug(f"Inapplicable command for buffering mode: {final}")
                        return True

                    # uppercase/lowercase and upper first letter are handled separately as they don't require any keypressing or clipboard pasting
                    if final == '<lc>':
                        self.caps_lock_state='lower'
                        return True

                    if final == '</lc>':
                        self.caps_lock_state='off'
                        return True

                    if final == '<uc>':
                        self.caps_lock_state='on'
                        return True

                    if final == '</uc>':
                        self.caps_lock_state='off'
                        return True
                    
                    if final == '<cc>':
                        self.shift_state='on'
                        return True

                    if final == '<ins>':
                        self.ins_state='on'
                        return True

                    if not self.dictaphone_mode and final == '<paste>':
                        pyperclip.copy(self.buffer)
                        keyboard.press_and_release(self.copypaste)
                        self.buffer = ""
                        self.buffering = False

                    # selection of text also has special treatement (select <n>)
                    #if final == '<select>'...

                    # if spell command encountered, make sure to not do any spacing and to take only first character
                    # TODO: check if it is better to take first letters or numbers from interims
                    if final == "<spell>":
                        self.spelling = True
                        self.spelling_started = True
                        return True
                    elif final == "</spell>":
                        self.spelling = False
                        self.spelling_started = None
                        return True
                    
                    # other commands are handled generically
                    # first check if for the key combination a sequence is defined. If yes, use pyautogui
                    # Otherwise use keyboard and clipboard as defined in settings
                    # NOTE: when in buffering mode, only commands that have the key 'copy2buffer' set in the settings, will be handled
                    
                    if final.lower() in self.settings["commands"]:
                        
                        # BUFFERING MODE
                        if self.buffering:
                            copy2buf = get_settings_value(self.settings, ["commands", final, self.dictation_type, "copy2buffer"])
                            if copy2buf:
                            # if self.settings["commands"][final][self.dictation_type]["copy2buffer"]:
                                self.buffer += self.settings["commands"][final][self.dictation_type]["copy2buffer"]
                        
                        # REGULAR MODE - ignore if in dictaphone mode
                        elif not self.dictaphone_mode:
                            if "keyboard_sequence" in self.settings["commands"][final][self.dictation_type]:
                                for key in self.settings["commands"][final][self.dictation_type]["keyboard_sequence"]:
                                    key_name = key.split('_')[0]
                                    action = key.split('_')[1]
                                    logger.debug(f"key: {key_name}, action: {action}")
                                    if action=="press":
                                        keyboard.press(key_name)
                                    elif action=="release":
                                        keyboard.release(key_name)
                                    elif action=="pressrelease":
                                        keyboard.press_and_release(key_name)
                                    time.sleep(self.settings["commands"][final][self.dictation_type]["keyboard_sequence_pause"])

                            if self.settings["commands"][final][self.dictation_type]["copyclip"]:
                                pyperclip.copy(self.settings["commands"][final][self.dictation_type]["copyclip"])
                            
                            if self.settings["commands"][final][self.dictation_type]["keyboard_press"]:
                                if self.settings["commands"][final][self.dictation_type].get("repeat", None):
                                    repeat = self.settings["commands"][final][self.dictation_type]["repeat"]
                                else:
                                    repeat = 1
                                for i in range(0, repeat):
                                    keyboard.press_and_release(self.settings["commands"][final][self.dictation_type]["keyboard_press"])

                        # remember if <nl> or <np> so that next time you start without a space before
                        # if 'capitalize-sentences' is on, fire shift_state
                        if final in ('<nl>', '<np>'):
                            logger.debug("Setting space flag to 'OFF'")
                            self.start_with_space=False 
                            if get_settings_value(self.settings, ['general', 'do-postprocessing', 'capitalize-sentences']):
                                self.shift_state = 'on'
                                logger.debug("<np>/<nl> encountered, setting shift state to 'ON'")
                    
                    return True

                # handle composite commands if not in dictaphone mode
                self.command_composite = False

                if not self.dictaphone_mode and "command" in hypothesis: 
                    logger.debug(f"::command composition detected")
                    self.command_composite = True
                    for i in range(0, int(hypothesis["command_argument"])):
                        if "keyboard_sequence" in self.settings["commands"][hypothesis["command"]][self.dictation_type]:
                            for key in self.settings["commands"][hypothesis["command"]][self.dictation_type]["keyboard_sequence"]:
                                key_name = key.split('_')[0]
                                action = key.split('_')[1]
                                logger.debug(f"key: {key_name}, action: {action}")
                                if action=="press":
                                    keyboard.press(key_name)
                                elif action=="release":
                                    keyboard.release(key_name)
                                elif action=="pressrelease":
                                    keyboard.press_and_release(key_name)
                                time.sleep(self.settings["commands"][hypothesis["command"]][self.dictation_type]["keyboard_sequence_pause"])

                        if self.settings["commands"][hypothesis["command"]][self.dictation_type]["copyclip"]:
                            pyperclip.copy(self.settings["commands"][hypothesis["command"]][self.dictation_type]["copyclip"])
                        
                        if self.settings["commands"][hypothesis["command"]][self.dictation_type]["keyboard_press"]:
                            if self.settings["commands"][hypothesis["command"]][self.dictation_type].get("repeat", None):
                                repeat = self.settings["commands"][hypothesis["command"]][self.dictation_type]["repeat"]
                            else:
                                repeat = 1
                            for i in range(0, repeat):
                                keyboard.press_and_release(self.settings["commands"][hypothesis["command"]][self.dictation_type]["keyboard_press"])

                    return True

                # if not command, take care for all processing:
                # - retain final if you want to process it later with the next one, e.g. 5000
                # - casing
                # - applying nl/np within final
                # - spelling
                # - postprocessing (replacements)
                # - detection of SPP codes
                
                # if spelling is on:
                # - take first letter only and make it uppercase
                # - if number written as word, convert it to number
                if self.spelling:
                    logger.debug("Črkovanje ...")
                    
                    if self.spelling_started:
                        self.spelling_started = False
                        if final[0]==" ":
                            final = f" {final[1].upper()}"
                        else:
                            final = final[0].upper()
                    else:
                        final = final.strip()[0].upper()
                    pyperclip.copy(final)

                # if INS is on, wait on a number identifier of a template
                elif self.ins_state == 'on':
                    if final.strip().isnumeric() and final.strip() in self.templates:
                        logger.debug(f"Pasting template with id = {final}")
                        pyperclip.copy(self.templates[final.strip()])
                    else:
                        logger.debug("Unknown template ID!")
                        final=""
                        pyperclip.copy("")
                    self.ins_state = 'off'

                else: # regular dictation

                    logger.debug("::regular segment detected")

                    # take care for commands captured as part of the final
                    final = re.sub(self._ptr_nl, "\n", final)
                    final = re.sub(self._ptr_np, "\n\n", final)
                    
                    # perform replacements 
                    if get_settings_value(self.settings, ["general", "do-postprocessing", "numbers"]):
                        final = self.perform_replacements(final)  

                    # format notes (user issues command "opomba" or konec "opombe") 
                    if get_settings_value(self.settings, ["general", "do-postprocessing", "format-notes"]):
                        final = self.format_notes(final) 

                    # perform spp code detection and extension 
                    if get_settings_value(self.settings, ["general", "do-postprocessing", "spp-codes"]): 
                        final = self.extend_spp_codes(final)  
    
                    # handle casing
                    logger.debug(f'CAPSLOCK: {self.caps_lock_state}, SHIFT: {self.shift_state}')
                    if self.caps_lock_state == 'on':
                        final = final.upper()
                        pyperclip.copy(final)
                        self.shift_state = 'off'
                        logger.debug(f"Sent to clipboard: {final}")
                    elif self.caps_lock_state == 'lower':
                        final = final.lower()
                        pyperclip.copy(final)
                        logger.debug(f"Sent to clipboard: {final.lower()}")
                    elif self.shift_state == 'on':
                        if final[0]!=' ':
                            final = final[0].upper()+final[1:]
                            pyperclip.copy(final)
                            logger.debug(f"Sent to clipboard: {final}")
                        else:
                            final = " " + final[1].upper()+final[2:]
                            pyperclip.copy(final)
                            logger.debug(f"Sent to clipboard:  {final}")
                        self.shift_state = 'off'
                    else:
                        pyperclip.copy(final)
                        logger.debug(f"Sent to clipboard: {final}")

                # if in dictaphone mode, do nothing, if in buffering mode, buffer the final, otherwise print to screen
                if self.buffering:
                    self.buffer += final
                elif not self.dictaphone_mode:
                    keyboard.press_and_release(self.copypaste)
                return True #NOTE: this was added to catch finals in the part of code where signals are emited ... 

    def create_translations_dict(self):
        """
        Extracts translations from settings and creates a dictionary
        """
        translations = dict()
        res = get_settings_value(self.settings, ['user-defined-commands', 'translations'])
        if res:
            for el in res:
                translations[el['source']]=el['target']
        return translations

    def get_translation(self, text):
        return self.translations.get(text.lower().strip(" .,!?"), None)

    @pyqtSlot()
    def run(self):

        try:
            # before starting dictation, move to previous window - expectation here is that
            # the toolbar got focus and that going back to previous window is the right thing to do.
            # If in dictaphone mode, this is not relevant
            if not self.dictaphone_mode and self.settings.get('general', False) and \
                self.settings['general'].get('autofocus', False) and \
                self.settings['general']['autofocus'].get('autofocus-with-key', False) and \
                self.settings['general']['autofocus'].get('autofocus-key', False):
                keyboard.press_and_release(self.settings['general']['autofocus']['autofocus-key'])

            # signal to the main GUI that the process has started
            self.signals.status.emit(dict(stype='progress', value='started'))
            # start the process
            res = self.stt.transcribe_from_microphone(continue_session=self.continue_session, keyListener=False)
            # in case streaming rised an error, res will be false and stt.connection_errors will hold the error log
            
            if not res:
                if len(self.stt.connection_errors)>0:
                    self.signals.result.emit(self.stt.connection_errors[-1])
                else:
                    self.signals.result.emit(dict(
                        messageType='ERROR',
                        message="Unknown error!"
                    ))
            else:
                
                # send ping time
                self.signals.status.emit(dict(stype='ping', value=f"{self.stt.elapsed_time_to_connect}"))

                # send sessionId to main GUI if needed for external commands
                # value in the next command is set only for consistency reasons. It could be read directly from the stt object
                self.signals.status.emit(dict(stype='sessionid', value=f"{self.stt.last_sessionId}"))
                
                while self.stt.session:
                    for value in self.stt.get_status(True):
                        res = self.get_text(value)
                        # get text only returns values for finals
                        if not res:
                            # if interim: show last n characters of the last interim
                            self.signals.status.emit(dict(stype='interim', value=f"{self.last_interim}..."[-_interim_display_len-3:]))
                        elif self.last_final in self.settings["legal-commands"][self.dictation_type] or self.command_composite:
                            # if final is one of the allowed commands, send signal to update GUI if neccessary and display as the final inetrim    
                            if self.last_final=='<showcmd>':
                                self.commands_helper_active=True
                            if self.last_final=='<close>':
                                self.commands_helper_active=False    
                            self.signals.status.emit(dict(stype='final-command', value=self.last_final))
                        else:
                            self.signals.status.emit(dict(stype='final', value=self.last_final))

                # if end button pressed before first final
                if not self.last_final:
                    self.signals.status.emit(dict(stype='progress', value='finished'))
                # if pause keyword
                elif self.last_final.lower().strip()==self.settings["user-defined-commands"]["pause-dictation"]:
                    self.signals.status.emit(dict(stype='progress', value='pause', buffer=self.buffer))
                # if end dictation keyword
                elif self.last_final.lower().strip()==self.settings["user-defined-commands"]["end-dictation"]:
                    # save current buffer to clipboard in case it was not pasted yet
                    if self.buffer:
                        pyperclip.copy(self.buffer)
                    self.signals.status.emit(dict(stype='progress', value='finished'))
                # all other endings are unexpected
                else:
                    logger.warning(f"Session ended unexpectedly or on button click with last interim: {self.last_interim} and last final: {self.last_final}")    
                result = self.stt.session_info
                self.signals.result.emit(result)  # Return the result of the processing
        except:
            traceback.print_exc()
            exctype, value = sys.exc_info()[:2]
            self.signals.error.emit((exctype, value, traceback.format_exc()))
        #else:
        #    self.signals.result.emit(result)  # Return the result of the processing
        finally:
            self.signals.finished.emit()  # Done

    def helper_window_closed(self):
        self.commands_helper_active = False

class MSMQSignals(QObject):
    """
    Defines the signals available from a running MSMQ thread.
    Supported signals are:
    - result: data received over MSMQ, format: dict(label, message)
    - error: tuple (exctype, value, traceback.format_exc() )
    - finished: No data
    """
    result = pyqtSignal(dict)
    error = pyqtSignal(tuple)
    finished = pyqtSignal() 

class MSMQ(QRunnable):
    """
    This class encapsulates all required functionality to send and receive mesages from other applications over MS message queue.
    An instance of the MSMQ class should be created when app starts...
    """
    def __init__(self, purge_before_start=False):
        super(MSMQ, self).__init__()
        self.signals = MSMQSignals()
        # open tb_inQueue for reading
        self.qinfo=win32com.client.Dispatch("MSMQ.MSMQQueueInfo")
        computer_name = os.getenv('COMPUTERNAME')
        self.qinfo.FormatName="direct=os:"+computer_name+"\\PRIVATE$\\tb_inQueue"
        self.queue=self.qinfo.Open(1,0)   # Open a ref to queue to read(1)
        # purge queue if required so
        if purge_before_start:
            self.purge_queue()
            logger.debug(f"Receiving queue has been purged!")
        self.legal_external_commands = [
            'login',
            'logout',
            'status',
            'start_dict',
            'stop_dict',
            'hide_toolbar',
            'show_toolbar',
            'pause',
            'continue',
            'start_buffering',
            'paste',
            'command_mode',
            'stop_command_mode',
        ]

    def send_message(self, message):
        pass

    def read_queue(self):
        msg=self.queue.Receive()
        return dict(
            label=msg.Label,
            body=msg.Body,
        )
    
    def view_queue(self, timeout):
        """
        Makes a peek into the Queue and if something is in it, it returns True
        The call is blocking but for max timeout*1000 miliseconds
        """
        return self.queue.Peek(pythoncom.Empty, pythoncom.Empty, timeout * 1000)

    def close_queue(self):
        self.queue.close()

    def purge_queue(self):
        """
        Deletes all messages from the queue
        """
        self.queue.Purge()

    @pyqtSlot()
    def run(self):
        
        self.listening = True
        try:
            while self.listening:
                if self.view_queue(timeout=1):
                    msg = self.read_queue()
                    logger.debug(f'Got external message: {msg["label"]} - {msg["body"]}')
                    self.signals.result.emit(msg) # Send received message to the main thread
        except:
            traceback.print_exc()
            exctype, value = sys.exc_info()[:2]
            self.signals.error.emit((exctype, value, traceback.format_exc()))
        finally:
            self.queue.close()
            logger.debug(f'Listener to external commands is down!')
            self.signals.finished.emit()

class VitasisLabel(QLabel):
    def __init__(self, parent):
        QLabel.__init__(self, parent)
        self.link = "https://vitasis.si/home"

    def mousePressEvent(self, event):
        webbrowser.open(self.link)

class CheckableComboBox(QComboBox):

    # Subclass Delegate to increase item height
    class Delegate(QStyledItemDelegate):
        def sizeHint(self, option, index):
            size = super().sizeHint(option, index)
            size.setHeight(20)
            return size

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # Make the combo editable to set a custom text, but readonly
        self.setEditable(True)
        self.lineEdit().setReadOnly(True)
        # Make the lineedit the same color as QPushButton
        palette = qApp.palette()
        palette.setBrush(QPalette.Base, palette.button())
        self.lineEdit().setPalette(palette)

        # Use custom delegate
        self.setItemDelegate(CheckableComboBox.Delegate())

        # Update the text when an item is toggled
        self.model().dataChanged.connect(self.updateText)

        # Hide and show popup when clicking the line edit
        self.lineEdit().installEventFilter(self)
        self.closeOnLineEditClick = False

        # Prevent popup from closing when clicking on an item
        self.view().viewport().installEventFilter(self)

    def resizeEvent(self, event):
        # Recompute text to elide as needed
        self.updateText()
        super().resizeEvent(event)

    def eventFilter(self, object, event):

        if object == self.lineEdit():
            if event.type() == QEvent.MouseButtonRelease:
                if self.closeOnLineEditClick:
                    self.hidePopup()
                else:
                    self.showPopup()
                return True
            return False

        if object == self.view().viewport():
            if event.type() == QEvent.MouseButtonRelease:
                index = self.view().indexAt(event.pos())
                item = self.model().item(index.row())

                if item.checkState() == Qt.Checked:
                    item.setCheckState(Qt.Unchecked)
                else:
                    item.setCheckState(Qt.Checked)
                return True
        return False

    def showPopup(self):
        super().showPopup()
        # When the popup is displayed, a click on the lineedit should close it
        self.closeOnLineEditClick = True

    def hidePopup(self):
        super().hidePopup()
        # Used to prevent immediate reopening when clicking on the lineEdit
        self.startTimer(100)
        # Refresh the display text when closing
        self.updateText()

    def updateText(self):
        texts = []
        for i in range(self.model().rowCount()):
            if self.model().item(i).checkState() == Qt.Checked:
                texts.append(self.model().item(i).text())
        text = ", ".join(texts)

        # Compute elided text (with "...")
        metrics = QFontMetrics(self.lineEdit().font())
        elidedText = metrics.elidedText(text, Qt.ElideRight, self.lineEdit().width())
        self.lineEdit().setText(elidedText)

    def addItem(self, text, data=None, checked=False):
        item = QStandardItem()
        item.setText(text)
        if data is None:
            item.setData(text)
        else:
            item.setData(data)
        item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
        if checked:
            item.setData(Qt.Checked, Qt.CheckStateRole)
        else:
            item.setData(Qt.Unchecked, Qt.CheckStateRole)
        self.model().appendRow(item)

    def addItems(self, texts, datalist=None):
        for i, text in enumerate(texts):
            try:
                data = datalist[i]
            except (TypeError, IndexError):
                data = None
            self.addItem(text, data)

    def currentData(self):
        # Return the list of selected items data
        res = []
        for i in range(self.model().rowCount()):
            if self.model().item(i).checkState() == Qt.Checked:
                res.append(self.model().item(i).data())
        return res

class Ui_MainWindow(QWidget, Ui_Toolbar):
    def __init__(
        self, 
        gui_object, 
        logger_folder=os.path.dirname(__file__)
        ):
        super().__init__()
        # ------------------------------------------------------------------------------------------------------    
        # windows sizing
        # ------------------------------------------------------------------------------------------------------
        self.main_window_height_S = scale(_main_window_height_S)
        self.main_window_height_L = scale(_main_window_height_L)
        self.main_window_width_S = scale(_main_window_width_S)
        self.main_window_width_L = scale(_main_window_width_L)
        # ------------------------------------------------------------------------------------------------------
        # init of Qsettings to store selected variables to registry
        # ------------------------------------------------------------------------------------------------------
        self.qt_settings = QSettings('Vitasis', 'TB-listener')
        # ------------------------------------------------------------------------------------------------------
        # init of the logger
        # ------------------------------------------------------------------------------------------------------
        self.logger_folder=logger_folder
        logger = setup_logging("toolbar", logger_folder)
        # ------------------------------------------------------------------------------------------------------
        # threading
        # ------------------------------------------------------------------------------------------------------
        self.threadpool = QThreadPool()
        logger.info("Multithreading with maximum %d threads" % self.threadpool.maxThreadCount())
        gui_object.setObjectName("MainWindow")
        # ------------------------------------------------------------------------------------------------------
        # init of common variables
        # ------------------------------------------------------------------------------------------------------
        ### common fonts
        self.bold_font = QtGui.QFont()
        self.bold_font.setBold(True)
        self.bold_font.setWeight(75)
        self.small_font = QtGui.QFont()
        self.small_font.setPointSize(7)
        self.large_font = QtGui.QFont()
        self.large_font.setPointSize(12)
        ### common icons
        self.icon_rec_idle = QtGui.QIcon()
        self.icon_rec_idle.addPixmap(QtGui.QPixmap(os.path.join(basedir, "record.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.icon_rec_started = QtGui.QIcon()
        self.icon_rec_started.addPixmap(QtGui.QPixmap(os.path.join(basedir, "recording.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.icon_rec_pause = QtGui.QIcon()
        self.icon_rec_pause.addPixmap(QtGui.QPixmap(os.path.join(basedir, "pause.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.icon_dict_idle = QtGui.QIcon()
        self.icon_dict_idle.addPixmap(QtGui.QPixmap(os.path.join(basedir, "dict.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.icon_dict_running = QtGui.QIcon()
        self.icon_dict_running.addPixmap(QtGui.QPixmap(os.path.join(basedir, "dict-red.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        
        # ------------------------------------------------------------------------------------------------------
        # Render of the designed GUI plus additional settings
        # ------------------------------------------------------------------------------------------------------
        self.setupUi(gui_object)
        #--------------------------------------------------------------------------------------------------------
        # add missing components, styling, actions and signals
        #--------------------------------------------------------------------------------------------------------
        ### extendable spacer to make QLabel for status info in the toolbar right-aligned
        spacer = QtWidgets.QWidget()
        spacer.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.toolBar.addWidget(spacer)
        ### Toolbar label "recording status" indicates buffering state
        self.label_rec_status = QtWidgets.QLabel()
        self.label_rec_status.setGeometry(QtCore.QRect(scale(0), scale(0), scale(40), scale(16)))
        self.label_rec_status.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        self.label_rec_status.setObjectName("label_rec_status")
        self.toolBar.addWidget(self.label_rec_status)
        ### Toolbar label "logged user" tells the current user name and number of initialized custom templetes
        self.label_logged_user = QtWidgets.QLabel()
        self.label_logged_user.setGeometry(QtCore.QRect(scale(0), scale(0), scale(40), scale(16)))
        self.label_logged_user.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        self.label_logged_user.setObjectName("label_logged_user")
        self.centralwidget.setStyleSheet(f"background-color: {cl_light_green_string}")
        self.toolBar.addWidget(self.label_logged_user)
        ### Toolbar icons
        icon_info = QtGui.QIcon()
        icon_info.addPixmap(QtGui.QPixmap(os.path.join(basedir, "info.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.actionInfo.setIcon(icon_info)
        icon_settings = QtGui.QIcon()
        icon_settings.addPixmap(QtGui.QPixmap(os.path.join(basedir, "settings.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.actionSettings.setIcon(icon_settings)
        icon_exit = QtGui.QIcon()
        icon_exit.addPixmap(QtGui.QPixmap(os.path.join(basedir, "exit_app3.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.actionExit.setIcon(icon_exit)
        self.actionAddWord.setIcon(self.icon_dict_idle)
        icon_dictaphone = QtGui.QIcon()
        icon_dictaphone.addPixmap(QtGui.QPixmap(os.path.join(basedir, "dictaphone.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.actionDictaphone.setIcon(icon_dictaphone)
        icon_debug = QtGui.QIcon()
        icon_debug.addPixmap(QtGui.QPixmap(os.path.join(basedir, "debug.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.pushButton_sendLogs.setIcon(icon_debug)
        icon_detailed_settings = QtGui.QIcon()
        icon_detailed_settings.addPixmap(QtGui.QPixmap(os.path.join(basedir, "settings.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.pushButton_detailed_settings.setIcon(icon_detailed_settings)
        ### label for interims - initialy empty string
        self.label_interim.setText("")
        ### model selection
        self.comboBox_model.activated.connect(self.model_config)
        self.comboBox_model.setEnabled(False)
        self.comboBox_model.setStyleSheet(f"background-color: {cl_light_green_string}")
        ### postprocessing options
        self.checkBox_punctuation.stateChanged.connect(lambda: self.change_postprocessing_feature('punct'))
        self.checkBox_capitalization.stateChanged.connect(lambda: self.change_postprocessing_feature('cap'))
        self.checkBox_denormalization.stateChanged.connect(lambda: self.change_postprocessing_feature('denorm'))
        self.checkBox_punctuation.setEnabled(False)
        self.checkBox_capitalization.setEnabled(False)
        self.checkBox_denormalization.setEnabled(False)
        ### radio buttons for destination format
        self.radioButton_plain.toggled.connect(self.select_output)
        self.radioButton_plain.setEnabled(False)
        self.radioButton_rtf.setChecked(True)
        self.radioButton_rtf.toggled.connect(self.select_output)
        self.radioButton_rtf.setEnabled(False)
        self.radioButton_docx.toggled.connect(self.select_output)
        self.radioButton_docx.setEnabled(False)
        self.radioButton_b21.toggled.connect(self.select_output)
        self.radioButton_b21.setEnabled(False)
        ### host
        self.label_host.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        self.lineEdit_host.setStyleSheet("background-color: white")
        ### username
        self.label_username.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        self.lineEdit_username.setStyleSheet("background-color: white")
        ## password
        self.label_password.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        self.lineEdit_password.setEchoMode(QtWidgets.QLineEdit.Password)
        self.lineEdit_password.returnPressed.connect(self.authenticate)
        self.lineEdit_password.setStyleSheet("background-color: white")
        ### destination window
        self.lineEdit_destination.setEnabled(False)
        ### multiselect comboBox for shares - can't be added via the designer
        self.comboBox_allowedUsers = CheckableComboBox(self.centralwidget)
        self.comboBox_allowedUsers.setGeometry(QtCore.QRect(scale(20), scale(580), scale(251), scale(20)))
        self.comboBox_allowedUsers.setObjectName("comboBox_allowedUsers")
        self.comboBox_allowedUsers.setEnabled(False)
        self.comboBox_allowedUsers.setStyleSheet(f"background-color: {cl_light_green_string}")
        ### input device selection
        self.comboBox_inputDevice.setEnabled(False)
        self.comboBox_inputDevice.currentTextChanged.connect(self.set_input_device)
        ### pushButton Login
        self.pushButton_authenticate.setDefault(True)
        self.pushButton_authenticate.setStyleSheet("background-color: lightgrey")
        self.pushButton_authenticate.clicked.connect(lambda: self.authenticate())
        ### pushButton Logout
        self.pushButton_logout.setEnabled(False)
        self.pushButton_logout.setStyleSheet("background-color: lightgrey")
        self.pushButton_logout.clicked.connect(lambda: self.logout())
        ### company Vitasis line at the bottom
        self.label_company = VitasisLabel(self.centralwidget)
        self.label_company.setGeometry(QtCore.QRect(scale(20), scale(680), scale(160), scale(16)))
        self.label_company.setFont(self.small_font)
        self.label_company.setObjectName("label_company")
        self.label_company.setText("@Vitasis 2022, info@vitasis.si")
        self.label_company.setCursor(QCursor(QtCore.Qt.PointingHandCursor))
        ### formatting
        self.pushButton_bold.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        self.pushButton_italic.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        self.pushButton_underline.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        self.pushButton_upper.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        self.pushButton_lower.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        self.pushButton_spell.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
        ### toolBar
        self.toolBar.setStyleSheet("background-color: lightblue")
        self.toolBar.setLayoutDirection(QtCore.Qt.LeftToRight)
        self.toolBar.setFloatable(True)
        self.actionRecord.setIcon(self.icon_rec_idle)
        self.actionRecord.setVisible(False)
        self.actionRecord.triggered.connect(self.open_dialog)
        self.recording_state = 'idle'
        self.actionSettings.triggered.connect(lambda: self.resize_settings_widget())
        self.actionExit.triggered.connect(lambda: self.exitApp())
        self.actionInfo.triggered.connect(lambda: self.open_commands_helper())
        self.actionAddWord.triggered.connect(lambda: self.open_add_word_dialog())
        #self.toolBar.setStyleSheet("QToolButton:!hover {background-color:lightblue} QToolBar {background: lightgrey}")
        # connect dictaphone button - initially hidden
        self.actionDictaphone.triggered.connect(lambda: self.dictaphone_mode())
        self.actionDictaphone.setVisible(False)
        # action to add words to dictionary is initially hidden. Not all models support dictionary management
        # when user authenticates or changes model, the action is rechecked and set to visible or hidden
        self.actionAddWord.setVisible(False)
        ### send button for logs
        self.pushButton_sendLogs.clicked.connect(lambda: self.open_send_logs())
        ### open detailed settings
        self.pushButton_detailed_settings.clicked.connect(lambda: self.open_detailed_settings())
        ### app version
        self.label_about.setText(f"TB-listener, verzija {app_version}")
        # ----------------------------------------------------------------------------------------
        # Init of not GUI atrtributes
        # ----------------------------------------------------------------------------------------
        ### user authentication status: {True, False}
        self.user_authenticated = False
        ### window handle to give focus to
        self.wtof = None
        ### variable to hold session name - will be used to patch session name when finished
        self.sessionName = None
        ### variable to hold session notes - will be used to patch session notes when finished
        self.session_notes = None
        ### init of the truebar BE object
        self.tb_current_config = None
        self.stt = None
        self.tb_params={
            "timeout": 2,
            "audio_format": truebar.pyaudio.paInt16,
            "audio_channels": 1,
            "audio_rate": 16000,
            "audio_chunk": int(16000 / 10)
        }
        self.stt=truebar.TruebarSTT(self.tb_params, self.logger_folder)
        self.stt_platform = None
        ### initial dictation type. Selected from {txt, rtf, docx, b21} 
        self.dictation_type = "rtf"
        ### init of the spp codes
        self.spp_codes = self.read_spp_codes()
        ### PID dialog status
        self.PID_dialog_opened = None
        ### commands helper dialog status
        self.COMMANDS_dialog_opened = None
        ### config initialization. TODO: settings are read again befor every new recording. Maybe redundant! 
        # At the same time, use-proxy settings will not be changed as it is set upon authentication and binded to the stt class
        self.settings = read_config()
        if not self.settings.get('general', True).get('suppress-notifications', True):
            self.qt_settings.setValue('suppress-notifications', False)
        ### a variable to hold streamer object when activated
        self.streamer = None
        ### initial buffer state - buffer is empty and buffering state is off
        self.buffer = ""
        self.buffering = False
        ### init of the listener for extenal commands - the listener will run in separate thread
        ### also, start queue for dispatching responses on external commands - no need for separate thread
        self.inQueue = None
        self.outQueue = None
        if get_settings_value(self.settings, ['integration', 'listen-for-external-commands']):
            self.start_command_listener()
            self.outQueue = self.start_outQueue()
        ### this attribute will hold identifier of the last processed external message and will reset to None after a response is send back over the outQueue
        self.active_external_msg_id = None
        ### init of templates if any in the template folder (check settings for where the template folder is)
        self.templates = self.read_templetes()
        # ----------------------------------------------------------------------------------------
        # Read saved values from the registry
        # ----------------------------------------------------------------------------------------
        self.model_updates = self.qt_settings.value('model-updates') or dict()
        self.init_fields()
        ### check settings if new version
        if self.new_version():
            try:
                self.merge_settings()
            except Exception as e:
                logger.error(f"Error when merging settings! Exception: {e}")
                
        QtCore.QMetaObject.connectSlotsByName(gui_object)

    def new_version(self):
        last_version = self.qt_settings.value('version-on-last-exit')
        if last_version and last_version != app_version:
            logger.warning(f"New version detected! Last version was {last_version}, new version is {app_version}!")
            return True
        else:
            return False

    def merge_settings(self):
        """
        Compares last settings with settings of the new version. The procedure is as follows:
        - if key is in both, new and old settings, and if it is specified as 'key to keep', replace new value with the old value
        - if key is in old settings only and at the same time it is annotated as 'key to keep', merge it into the new settings
        Keys to keep are specified as a list of values under 'keys-to-keep' key in the settings. Every key that is included in
        this list must be taken or merged from the old settings - i.e. must override new values.
        NOTE: if 'keys-to-keep' key is not specified in new settings, it will be initialized as an empty list - meaning that
        all old keys will be ignored.
        """
        def recursive_merge(new_d, old_d, keys_to_keep):
            for k in old_d.keys():
                if k in keys_to_keep:
                    if new_d[k]!=old_d[k]: 
                        new_d[k]=old_d[k]
                        logger.debug(f"Merging old key {k}.")
                elif isinstance(old_d[k], dict) and k in new_d: 
                    new_d[k]=recursive_merge(new_d[k], old_d[k], keys_to_keep)
            return new_d

        if os.path.isfile(os.path.join(os.getenv('APPDATA'), 'vitasis', 'tb-listener', f'settings.yaml')):
            old_settings=read_config(os.path.join(os.getenv('APPDATA'), 'vitasis', 'tb-listener', 'settings.yaml'))
            new_settings=read_config(os.path.join(basedir, "settings.yaml"))     
            new_keys=[k for k in new_settings.keys() if k not in old_settings]
            missing_keys=[k for k in old_settings.keys() if k not in new_settings]
            common_keys_different_values=[k for k in new_settings.keys() if k in old_settings and new_settings[k]!=old_settings[k]]
            keys_to_keep=new_settings.get('keys-to-keep', False)
            logger.debug(f"Settings comparison results: new keys: {len(new_keys)}, missing keys: {len(missing_keys)}, common keys with different values: {len(common_keys_different_values)}")
            logger.debug(f"Keys to keep: {keys_to_keep}")

            loaded_settings = copy.deepcopy(self.settings)
            recursive_merge(new_settings, old_settings, keys_to_keep)
            
            # if settings changed due to the merge, save old settings into settings_{version}.yaml and 
            # merged settings into settings.yaml 
            if loaded_settings != new_settings:
                last_version = self.qt_settings.value('version-on-last-exit') or time.strftime("%Y%m%d_%H%M%S", time.localtime())
                shutil.copyfile(
                    os.path.join(user_appdata, "settings.yaml"),
                    os.path.join(user_appdata, f'settings_{last_version}.yaml')
                    )
                save_config(new_settings)
                self.settings=new_settings
                logger.debug(f"New settings have been merged and saved to disk.")
        else:
            logger.debug(f"Old settings can't be found. New settings will be used without changes.")

    def init_fields(self):
        #print(self.qt_settings.allKeys())
        if 'autofocus-window-title' in self.qt_settings.allKeys():
            self.lineEdit_destination.setText(self.qt_settings.value('autofocus-window-title'))
        elif get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-by-window-title']):
            wtof = get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-window-title'])
            if wtof:
                self.lineEdit_destination.setText(wtof)

        self.lineEdit_host.setText(self.qt_settings.value('host'))
        self.lineEdit_username.setText(self.qt_settings.value('username'))
        
        if self.qt_settings.value('output_format')=='plain':
            self.radioButton_plain.setChecked(True)
        elif self.qt_settings.value('output_format')=='rtf':
            self.radioButton_rtf.setChecked(True)
        elif self.qt_settings.value('output_format')=='docx':
            self.radioButton_docx.setChecked(True)
        elif self.qt_settings.value('output_format')=='b21':
            self.radioButton_b21.setChecked(True)

    def hotkey(self, action):
        # if commands are currently opened, do nothig
        if self.COMMANDS_dialog_opened:
            return

        if action=='commands-help':
            self.open_commands_helper()

        if action=='toggle':
            #print(self.recording_state, "*******************************")
            # if currently PID dialog opened, do nothing
            if not self.PID_dialog_opened:
                # if in idle and user authenticated -> open dialog to enter PID
                if self.recording_state=='idle' and self.user_authenticated:
                    self.actionRecord.trigger()
                # if in pause -> continue recording
                elif self.recording_state=='pause':
                    # set self.wtof to None to indicate that the command is not comming from an external command
                    self.wtof = None
                    self.continue_sessionId = None
                    self.record()
                # if in recording -> go to pause mode
                elif self.recording_state=='recording':
                    self.pause_recording()

    def read_templetes(self):
        """
        Reads templates from template folder (the template folder is defined in settings)
        TODO: Only plain text is currently supported. Add support for rtf, html and md
        """
        templates = dict()
        if os.path.isdir(os.path.join(basedir, "templates")):
            for file in os.listdir(os.path.join(basedir, "templates")):
                # NOTE: currently, only plain txt is supported
                if file[0] not in ('.', '~') and file.split(".")[-1] in ('txt', 'rtf', 'html', 'md'):
                    with open(os.path.join(os.path.join(basedir, "templates"), file), "r", encoding="utf-8") as ftemplate:
                        templates[file.split(".")[0]] = "".join(ftemplate.readlines())
            logger.debug(f"{len(templates)} templates imported from {os.path.join(basedir, 'templates')}")
        return templates
        
    def start_command_listener(self):
        logger.info("Starting listener for external commands!")
        self.inQueue = MSMQ(
            purge_before_start=get_settings_value(self.settings, ['integration', 'purge-queue-on-start'])
            )
        self.inQueue.signals.result.connect(self.external_command_received)
        self.inQueue.signals.finished.connect(self.queue_listener_stopped)
        # execute
        self.threadpool.start(self.inQueue)
        return
    
    def start_outQueue(self):
        # open tb_outQueue for writing
        queue=win32com.client.Dispatch("MSMQ.MSMQQueueInfo")
        computer_name = os.getenv('COMPUTERNAME')
        queue.FormatName="direct=os:"+computer_name+"\\PRIVATE$\\tb_outQueue"
        try:
            q = queue.Open(2,0)   # Open a ref to queue to write(2)
            return q
        except Exception as e:
            logger.warning(f"Can't open out queue for sending responses to exteral systems.\nIntegration over MSQM will not work.")
            return None

    def read_spp_codes(self):
        spp = dict()
        if os.path.isfile(os.path.join(basedir, "spp_codes.csv")):
            with open(os.path.join(basedir, "spp_codes.csv"), "r", encoding="utf-8") as fspp:
                for line in fspp.readlines():
                    fields = line.split("|")
                    spp[fields[1]]=fields[2].strip()
            return spp
        else:
            logger.error(f"File gui/spp_codes.csv is missing! SPP codes will not be detected")
            return None
           
    def model_config(self):
        """
        If selected model is different than the current one,
        change the configuration. Otherwise do nothing.
        """
        # selected model
        selected_model = self.comboBox_model.currentText().split(",")

        # check if selected model is different than current and in case
        # it is, change the servis configuration
        current_framework = self.tb_current_config["stt"]["framework"]["value"] 
        current_language = self.tb_current_config["stt"]["language"]["value"] 
        current_domain = self.tb_current_config["stt"]["domain"]["value"]
        current_version = self.tb_current_config["stt"]["model"]["value"]
        
        if not (
            selected_model[0].strip()==current_framework and
            selected_model[1].strip()==current_language and
            selected_model[2].strip()==current_domain and
            selected_model[3].strip()==current_version
            ):
            self.stt.set_model(
                selected_model[0].strip(),
                selected_model[1].strip(),
                selected_model[2].strip(),
                selected_model[3].strip(),
            )
            self.tb_current_config = self.stt.get_configuration()
            # Enable dictionary if available
            self.dict_phoneset = self.is_dict_available(
                language=selected_model[1].strip(),
                domain=selected_model[2].strip(),
                model=selected_model[3].strip(),
                )
            self.actionAddWord.setVisible(self.dict_phoneset is not False)
            # set postprocessing features
            self.set_postprocessing_features()

    def dictaphone_mode(self):
        if self.actionDictaphone.isChecked():
            self.label_interim.setText("Vklopljen je diktafonski način!")
            self.widget_status.setStyleSheet(f"background-color: {cl_light_yellow_string}")
        else:
            self.label_interim.setText("")
            self.widget_status.setStyleSheet(f"background-color: {cl_light_green_string}")

    def is_dict_available(self, language, domain, model):
        if self.stt.get_model_phoneset(
            language=language,
            domain=domain,
            model=model,
            ):
            return True
        else:
            return False
            
    def set_input_device(self):
        self.stt.selected_input_device_index = self.comboBox_inputDevice.currentIndex()
        logger.debug(f"Device input set to {self.comboBox_inputDevice.currentIndex()} - {self.comboBox_inputDevice.currentText()}")

    def exitApp(self):

        # display info on open threads
        logger.debug(f"Number of running threads: {threading.active_count()}")
        current_thread = threading.current_thread()
        #print(self.threadpool.activeThreadCount())
        logger.debug(f"Currently active thread: OS-ID={current_thread.native_id}, IDENT={current_thread.ident}, NAME={current_thread.getName()}")
        logger.debug(f"Current state: {self.recording_state}")
        logger.debug(f"Buffering state: {self.buffering}")

        # if currently in recording mode, do nothing
        if self.recording_state=='recording':
            logger.debug(f"Exit command will be ignored - the system is in recording state!")
            return

        # if in pause mode, don't exit. Patch session name and shares and go to "idle" mode
        if self.recording_state=='pause':
            # patch session name
            if self.sessionName:
                self.stt.patch_session_name(sessionid=self.stt.last_sessionId, name=self.sessionName)
                self.sessionName = None
            elif self.sessionName is None:
                logger.warning(f"Session {self.stt.last_sessionId} has no name defined and will not be patched!")
            self.actionRecord.setIcon(self.icon_rec_idle)
            self.recording_state='idle'
            # patch session shares
            users2share = self.comboBox_allowedUsers.currentText()
            if users2share != "":
                for u in users2share.split(','):
                    userid = self.get_userid(u)
                    if not userid or not str(userid).isnumeric():
                        logger.warning(f"Cannot add user {u} to session {self.stt.last_sessionId}. User with this username does not exist in your group.")
                        continue
                    self.stt.add_session_shares(sessionid=self.stt.last_sessionId, userid=int(userid))
            # return in idle mode
            return

        # save settings
        self.qt_settings.setValue('auto-punctuation', self.checkBox_punctuation.isChecked())
        self.qt_settings.setValue('auto-capitalization', self.checkBox_capitalization.isChecked())
        self.qt_settings.setValue('denormalization', self.checkBox_denormalization.isChecked())
        self.qt_settings.setValue('autofocus-window-title', self.lineEdit_destination.text())
        self.qt_settings.setValue('username', self.lineEdit_username.text())
        self.qt_settings.setValue('host', self.lineEdit_host.text())
        self.qt_settings.setValue('allowedUsers', self.comboBox_allowedUsers.currentText())
        self.qt_settings.setValue('input-device', self.comboBox_inputDevice.currentText())
        if self.radioButton_plain.isChecked():
            self.qt_settings.setValue('output_format', 'plain')
        elif self.radioButton_rtf.isChecked():
            self.qt_settings.setValue('output_format', 'rtf')
        elif self.radioButton_docx.isChecked():
            self.qt_settings.setValue('output_format', 'docx')
        elif self.radioButton_b21.isChecked():
            self.qt_settings.setValue('output_format', 'b21')    
        self.qt_settings.setValue('window size', self.size())
        self.qt_settings.setValue('window position', self.pos())

        # signal listener of external commands to stop listening on the MSMQ
        # closing queue while MSMQ runner is still listening will raise exception. Couldn't find any other solution. 
        if self.inQueue:
            self.inQueue.listening = False
            #logger.info(f"Closing listener for external commands!")
            #self.inQueue.queue.close()    
            # close outQueue as well
            logger.info(f"Closing outQueue!")
            self.outQueue.close()

        # save settings.yaml to AppData location as a backup
        # NOTE: this is now redundant, since settings are saved to this location every time they change
        #appdata_folder = os.path.join(os.getenv('APPDATA'), 'vitasis', 'tb-listener')
        #shutil.copy(os.path.join(basedir, "settings.yaml"), os.path.join(appdata_folder, f"settings.yaml"))
        # save date-time and app version on exit
        self.qt_settings.setValue('last-exit', time.strftime("%d.%m. %Y %H:%M:%S", time.localtime()))
        self.qt_settings.setValue('version-on-last-exit', app_version)

        sys.exit()

    def resize_settings_widget(self):
        if self.frameGeometry().height() < self.main_window_height_L:
            # first switch off dictaphone mode if ON 
            self.actionDictaphone.setChecked(False)
            self.dictaphone_mode()
            # now resize
            self.setFixedSize(self.main_window_width_S, self.main_window_height_L)
        else:
            self.setFixedSize(self.main_window_width_L, self.main_window_height_S)
       
    def logout(self):
        self.lineEdit_username.setText("")
        self.lineEdit_password.setText("")
        self.comboBox_model.clear()
        self.pushButton_authenticate.setEnabled(True)
        self.pushButton_logout.setEnabled(False)
        self.lineEdit_destination.setEnabled(False)
        self.comboBox_allowedUsers.setEnabled(False)
        self.comboBox_inputDevice.setEnabled(False)
        self.radioButton_plain.setEnabled(False)
        self.radioButton_rtf.setEnabled(False)
        self.radioButton_docx.setEnabled(False)
        self.radioButton_b21.setEnabled(False)
        self.checkBox_punctuation.setEnabled(False)
        self.checkBox_capitalization.setEnabled(False)
        self.checkBox_denormalization.setEnabled(False)
        self.pushButton_sendLogs.setEnabled(False)
        self.pushButton_detailed_settings.setEnabled(False)
        self.lineEdit_host.setStyleSheet("background-color: white")
        self.lineEdit_username.setStyleSheet("background-color: white")
        self.lineEdit_password.setStyleSheet("background-color: white")
        self.comboBox_model.setStyleSheet(f"background-color: {cl_light_green_string}")
        self.comboBox_allowedUsers.setStyleSheet(f"background-color: {cl_light_green_string}")
        self.comboBox_inputDevice.setStyleSheet(f"background-color: {cl_light_green_string}")
        self.lineEdit_destination.setStyleSheet(f"background-color: {cl_light_green_string}")
        self.user_authenticated=False
        self.actionRecord.setVisible(False)
        self.actionDictaphone.setVisible(False)
        self.actionAddWord.setVisible(False)
        self.counter = 0
        self.label_interim.setText("")
        self.label_logged_user.setText(" ")
        self.lineEdit_username.setFocus()

    def get_userid(self, username):
        if self.stt:
            for u in self.stt.group_users:
                if username.strip() == u['username'].strip():
                    return u['id']
        return None

    def set_postprocessing_features(self):
        if get_settings_value(self.tb_current_config, ['nlp', 'punctuation', 'enabled', 'isAllowed']):
            self.checkBox_punctuation.setEnabled(True)
            self.checkBox_punctuation.setChecked(get_settings_value(self.tb_current_config, ['nlp', 'punctuation', 'enabled', 'value']))
        else: 
            self.checkBox_punctuation.setEnabled(False)
        
        if get_settings_value(self.tb_current_config, ['nlp', 'enableTruecasing', 'isAllowed']):
            self.checkBox_capitalization.setEnabled(True)
            self.checkBox_capitalization.setChecked(get_settings_value(self.tb_current_config, ['nlp', 'enableTruecasing', 'value']))
        else: 
            self.checkBox_capitalization.setEnabled(False)
        
        if get_settings_value(self.tb_current_config, ['nlp', 'enableDenormalization', 'isAllowed']):
            self.checkBox_denormalization.setEnabled(True)
            self.checkBox_denormalization.setChecked(get_settings_value(self.tb_current_config, ['nlp', 'enableDenormalization', 'value']))
        else: 
            self.checkBox_denormalization.setEnabled(False)

    def change_postprocessing_feature(self, feature):
        if self.user_authenticated:
            if feature=='punct':
                new_config = dict(nlp=dict(punctuation=dict(enabled=dict(value=self.checkBox_punctuation.isChecked()))))
                self.stt.update_configuration(json.dumps(new_config))
            elif feature=='cap':
                new_config = dict(nlp=dict(enableTruecasing=dict(value=self.checkBox_capitalization.isChecked())))
                self.stt.update_configuration(json.dumps(new_config))
            elif feature=='denorm':
                new_config = dict(nlp=dict(enableDenormalization=dict(value=self.checkBox_denormalization.isChecked())))
                self.stt.update_configuration(json.dumps(new_config))

    def authenticate(self):
        def get_last_input_device_index():
            """
            Reads all inputs and returns index of the input that was selected last time
            If such input is not found, it returns zero, which refers to the System Default Input
            """
            last_input_device = self.qt_settings.value('input-device')
            if last_input_device:
                for k in self.stt.input_devices.keys():
                    if last_input_device.lower() == self.stt.input_devices[k].lower():
                        self.stt.selected_input_device_index = k
                        return k
            # if last_input_device not found among available input devices or last_input_device is None, 
            # set device_index to zero indicating that system input should be used 
            self.stt.selected_input_device_index = 0
            return 0 
            
        # create config object
        _PLATFORM = self.lineEdit_host.text()
        use_proxy = get_settings_value(self.settings, ['proxy-server', 'use-proxy-server']) 
        proxy_ip = get_settings_value(self.settings, ['proxy-server', 'proxy-host-ip'])
        proxy_port = get_settings_value(self.settings, ['proxy-server', 'proxy-host-port'])
        
        _PROXY = None
        if use_proxy and proxy_ip and proxy_port:
            _PROXY = f"{proxy_ip}:{proxy_port}"

        self.tb_params.update(
            ASR_server=_PLATFORM+"-api.true-bar.si",
            username=self.lineEdit_username.text(),
            password=self.lineEdit_password.text(),
            proxy=_PROXY,
        )
        self.tb_params["auth-url"]="https://"+_PLATFORM+"-auth.true-bar.si/auth/realms/truebar/protocol/openid-connect/token"

        # create Truebar object to work with
        #self.stt=truebar.TruebarSTT(self.tb_params, self.logger_folder)

        # get authentication token
        if not self.stt.get_auth_token():
            if 'status_code' in self.stt.connection_errors[-1] and self.stt.connection_errors[-1]['status_code']==401:
                self.lineEdit_username.setStyleSheet("border: 1px solid red;")
                self.lineEdit_password.setStyleSheet("border: 1px solid red;")
            else:
                self.lineEdit_host.setStyleSheet("border: 1px solid red;")
            self.pushButton_authenticate.setEnabled(True)
            self.pushButton_logout.setEnabled(False)
            self.user_authenticated = False
            self.label_logged_user.setText(" ")

            return False

        else:
            logger.info(f"User '{self.lineEdit_username.text()}' authenticated on host '{self.lineEdit_host.text()}'")
            logger.info(f"Client IP: {self.stt.get_ip()}")
            logger.info(f"Cleint MAC: {hex(uuid.getnode())}")
            # populate combo for the selection of an input device and preselect microphone 
            # that was used last time - if still available, otherwise select system input
            # NOTE: index of the first input device, detected by the system, starts with 1 and not 0!!! 
            # Zero index is used in this code for System Default Input
            self.comboBox_inputDevice.addItem("Privzet vhod sistema") # this goes to index 0
            for el in self.stt.input_devices.keys():
                self.comboBox_inputDevice.addItem(self.stt.input_devices[el])
            self.comboBox_inputDevice.setCurrentIndex(get_last_input_device_index())

            self.pushButton_authenticate.setEnabled(False)
            self.pushButton_logout.setEnabled(True)
            self.lineEdit_username.setStyleSheet("")
            self.lineEdit_password.setStyleSheet("")
            self.lineEdit_host.setStyleSheet("")
            self.lineEdit_destination.setStyleSheet("background-color: white")
            self.comboBox_model.setStyleSheet("background-color: white")
            self.comboBox_allowedUsers.setStyleSheet("background-color: white")
            self.comboBox_inputDevice.setStyleSheet("background-color: white")
            # show icons
            self.actionRecord.setVisible(True)
            self.actionDictaphone.setVisible(True)
            self.user_authenticated = True
            self.label_logged_user.setText(f"T: {len(self.templates)} | U: {self.lineEdit_username.text()}  ")
            # enable settings
            self.lineEdit_destination.setEnabled(True)
            self.comboBox_allowedUsers.setEnabled(True)
            self.radioButton_plain.setEnabled(True)
            self.radioButton_rtf.setEnabled(True)
            self.radioButton_docx.setEnabled(True)
            self.radioButton_b21.setEnabled(True)
            self.comboBox_model.setEnabled(True)
            self.comboBox_inputDevice.setEnabled(True)
            self.checkBox_punctuation.setEnabled(True)
            self.checkBox_capitalization.setEnabled(True)
            self.checkBox_denormalization.setEnabled(True)
            self.pushButton_sendLogs.setEnabled(True)
            self.pushButton_detailed_settings.setEnabled(True)
            # get available models
            available_models=self.stt.get_available_models()
            for model in available_models:
                self.comboBox_model.addItem(f"{model['framework']}, {model['language']}, {model['domain']}, {model['model']}")
            # read current configuration
            self.tb_current_config = self.stt.get_configuration()
            # set combobox selection to the current model
            for i in range(self.comboBox_model.count()):
                if self.comboBox_model.itemText(i)==f"{self.tb_current_config['stt']['framework']['value']}, {self.tb_current_config['stt']['language']['value']}, {self.tb_current_config['stt']['domain']['value']}, {self.tb_current_config['stt']['model']['value']}":
                    self.comboBox_model.setCurrentIndex(i)
                    break

            logger.info(f"Selected model: {self.comboBox_model.currentText()}")
            logger.info(f"Selected input device: {self.comboBox_inputDevice.currentText()}")

            # Enable dictionary if supported by the model
            self.dict_phoneset = self.is_dict_available(
                    language=self.tb_current_config['stt']['language']['value'],
                    domain=self.tb_current_config['stt']['domain']['value'],
                    model=self.tb_current_config['stt']['model']['value'],
                    )
            self.actionAddWord.setVisible(self.dict_phoneset is not False)
            # check status of the model - if running, color the icon in red and
            # start timer to periodically check the status! Once finished, stop
            # the timer
            if get_model_update_status(
                language=self.tb_current_config['stt']['language']['value'],
                domain=self.tb_current_config['stt']['domain']['value'],
                model=self.tb_current_config['stt']['model']['value'],
                tb_object=self.stt,
                )=='RUNNING':
                self.actionAddWord.setIcon(self.icon_dict_running)
                self.model_update_timer=QTimer()
                self.model_update_timer.setInterval(3000)
                self.model_update_timer.timeout.connect(self.check_model_status)
                self.model_update_timer.start()
            else:
                self.actionAddWord.setIcon(self.icon_dict_idle)
            # Set postprocessing as it was the last time
            self.set_postprocessing_features()
            # Get group users
            self.stt.group_users = self.stt.get_users_in_my_group()
            users2share = []
            if 'allowedUsers' in self.qt_settings.allKeys():
                users2share = self.qt_settings.value('allowedUsers')
                if users2share:
                    users2share = [usr.strip() for usr in users2share.split(',')]
                else:
                    users2share = []
            for el in self.stt.group_users:
                self.comboBox_allowedUsers.addItem(el['username'], None, el['username'] in users2share)
            self.comboBox_allowedUsers.updateText()
            # Resize and move window to last position
            self.show_interims()
            if 'window position' in self.qt_settings.allKeys():
                self.move(self.qt_settings.value('window position'))
            # if minimize-after-login is True, minimize the toolbar into the tray
            if get_settings_value(self.settings, ['general', 'system-tray', 'minimize-after-login']):
                self.hide()
            
            return True

    def check_model_status(self):
        language = self.tb_current_config["stt"]["language"]["value"]
        domain = self.tb_current_config["stt"]["domain"]["value"]
        model = self.tb_current_config["stt"]["model"]["value"]
        
        model_updater_status = get_model_update_status(language=language, domain=domain, model=model, tb_object=self.stt)

        # refresh registry entry 
        self.model_updates[f"{language}{domain}{model}"].update(
            status=model_updater_status,
            last_check=time.strftime('%H:%M:%S', time.gmtime(time.time()))
            )
        self.qt_settings.setValue('model-updates', self.model_updates)
        
        if model_updater_status:
            # if updater is running, keep icon in red
            if model_updater_status == "RUNNING":
                self.actionAddWord.setIcon(self.icon_dict_running)
            # if update is IDLE or in FAILED state, put icon back in black and stop timer if running
            else:
                self.actionAddWord.setIcon(self.icon_dict_idle)
                if self.model_update_timer.isActive():
                    self.model_update_timer.stop()

    def record(self):
        if self.recording_state == 'recording':
            self.stop_recording()
            self.recording_state = 'idle'
        else:
            self.start_recording()

    def stop_recording(self):
        self.actionRecord.setIcon(self.icon_rec_idle)
        logger.debug(f"Recording status: {self.recording_state}")
        logger.info("Recording stopped")
        self.stt.end_session()
        # reset buffer
        self.buffering = False
        self.buffer = ""

    def pause_recording(self):
        self.recording_state = "pause"
        self.actionRecord.setIcon(self.icon_rec_pause)
        logger.debug(f"Recording status: {self.recording_state}")
        logger.info("Recording paused")
        self.stt.end_session()

    def show_interims(self):
        self.setFixedSize(self.main_window_width_L, self.main_window_height_S)

    def display_status(self, status):
        # status type 'progress' brings message on the streamer status, which can be one of 
        # the following: 'started' - means that recording has started, 'pause' or 'finished'

        if status["stype"]=="progress":
            if status["value"]=="started":
                self.actionRecord.setIcon(self.icon_rec_started)
                self.recording_state = 'recording'
            elif status["value"]=="finished":
                self.actionRecord.setIcon(self.icon_rec_idle)
                self.recording_state = 'idle'
                self.label_rec_status.setText("")
            elif status["value"]=="pause":
                self.actionRecord.setIcon(self.icon_rec_pause)
                self.recording_state = 'pause'
                self.buffer = status["buffer"]
        
        # label_logged_user is abused to additionaly display ping time (P) and number of templates (T)
        elif status["stype"]=="ping":
            self.label_logged_user.setText(f"P: {int(float(status['value'])*1000)} ms | T: {len(self.templates)} | U: {self.lineEdit_username.text()}  ")
        
        # status type 'sessionid' will come when session identifier has been assigned to the session.
        # If an external listener is activated, this message is forwarded to the out queue 
        elif status["stype"]=="sessionid":
            if self.active_external_msg_id:
                #self.send_reponse(self.active_external_msg_id, self.stt.last_sessionId)
                self.send_reponse(self.active_external_msg_id, f'cmd_status=True | response={self.stt.last_sessionId}')
        
        # with status type 'interim' we intercept ASR interims and display them in the toolbar
        # In dictaphone mode this message is ignored
        elif status["stype"]=="interim" and not self.actionDictaphone.isChecked():
            self.label_interim.setStyleSheet("color: gray")
            self.label_interim.setText(status["value"])
        
        elif status["stype"]=="final-command":
            # if formatting command, make neccessary changes on GUI
            if status["value"] in ('<showcmd>', '<close>', '<b>', '</b>', '<i>', '</i>', '<u>', '</u>', '<uc>', '</uc>', '<lc>', '</lc>', '<spell>', '</spell>'):
                self.toggle_button(status["value"], status["value"][1]!="/")
                # if open/close commands dialog ...
                if status["value"]=='<showcmd>':
                    self.open_commands_helper()
                if status["value"]=='<close>':
                    self.commands_helper.ui.pushButton_close.click()
                    self.COMMANDS_dialog_opened=False
                    wtof = self.lineEdit_destination.text().strip()
                    if wtof:
                        set_focus_to_window(wtof)
                # if UC, switch off LC and vice versa
                if status["value"]=='<uc>':
                    self.toggle_button('<lc>', False)
                if status["value"]=='<lc>':
                    self.toggle_button('<uc>', False)
            
            # if command to start buffering, indicate that on the GUI
            if status["value"]==get_settings_value(self.settings, ["user-defined-commands", "start-buffering"]):
                self.label_rec_status.setText("\u25BA pomnjenje | ")
                self.buffering = True
                #self.flashing_flag = True

            # if command to end buffering, clear the label indicating that on the GUI
            if status["value"]=="<paste>":
                self.label_rec_status.setText("")
                self.buffering = False
                # setting flashing flag to False is actually redundant since there is another condition that 
                # has to be met in order for flashing to work, namely the system must be in "buffering" mode...
                #self.flashing_flag = False

            # show the command as last interim (if not in dictaphone mode). Use black color
            if not self.actionDictaphone.isChecked():
                self.label_interim.setStyleSheet("color: black")
                self.label_interim.setText(f'<b>ukaz</b>: {status["value"].replace(">","").replace("<","")}')
            
            if get_settings_value(self.settings, ['general', 'beep-on-finals']): 
                winsound.Beep(4000, 5)
        
        elif status["stype"]=="final":
            # show fist n characters of the final as last interim (unless in dictaphone mode)
            if not self.actionDictaphone.isChecked():
                self.label_interim.setStyleSheet("color: black")
                if len(status["value"]) > _interim_display_len:
                    show_text = "..." + status["value"][-_interim_display_len:]
                else:
                    show_text = status["value"]
                self.label_interim.setText(f'<b>{show_text}</b>')
            if get_settings_value(self.settings, ['general', 'beep-on-finals']):
                winsound.Beep(4000, 10)

    def print_output(self, res):
        # If something went wrong in the TB library, the result will include key 'messageType' with value 'ERROR' and 
        # key 'message' with the description of the error 
        if 'messageType' in res and res['messageType']=='ERROR':
            logger.error(f"Error in streaming audio. Error message from server: {res['message']}")
            self.label_interim.setStyleSheet(f"color: {cl_message_failed}")
            self.label_interim.setText(f"Napaka pri povezavi na strežnik!")
            self.recording_state = 'idle'
            self.actionRecord.setIcon(self.icon_rec_idle)
            # if session start was triggered from an external command, send response
            if self.active_external_msg_id:
                 self.send_reponse(self.active_external_msg_id, f'cmd_status=False | response=Error in streaming audio. Error message: {res["message"]}')

        # Otherwise the results will hold session info
        else:  
            logger.debug(f"SESSION INFO: {res}")

    def thread_complete(self):
        logger.debug("THREAD COMPLETE!")
        # reset buttons
        for b in ('<b>', '<i>', '<u>', '<uc>', '<lc>', '<spell>'):
            self.toggle_button(b, on=False)

        # if session has been dropped, do nothing
        if self.stt.state == 'Error':
            logger.warning(f"Session in error state - session name and shares will not be patched!")
            return False

        # else if sessionId is not None, patch the session
        elif self.stt.last_sessionId:
            # patch session name and notes if session finished (in idle, not in pause)
            if (self.sessionName or self.session_notes) and self.recording_state=='idle':
                self.stt.patch_session_name(sessionid=self.stt.last_sessionId, name=self.sessionName, notes=self.session_notes)
                self.sessionName = None
                self.session_notes = None
            elif self.recording_state=='idle':
                logger.warning(f"Session {self.stt.last_sessionId} has no name or notes and will not be patched!")
            
            # add user shares if session finished (in idle, not in pause) 
            if self.recording_state=='idle':
                users2share = self.comboBox_allowedUsers.currentText()
                if users2share != "":
                    for u in users2share.split(','):
                        userid = self.get_userid(u)
                        if not userid or not str(userid).isnumeric():
                            logger.warning(f"Cannot add user {u} to session {self.stt.last_sessionId}. User with this username does not exist in your group.")
                            continue
                        self.stt.add_session_shares(sessionid=self.stt.last_sessionId, userid=int(userid))
            return True

    def handle_streaming_errors(self, error):
        # This will happen when an error will occur in the Streamer thread, other then the TB error
        logger.error(f"Error in the STREAMER thread. {error}")
        self.label_interim.setStyleSheet(f"color: {cl_message_failed}")
        self.label_interim.setText(f"Napaka pri izvajanju procesa razpoznave!")
        self.recording_state = 'idle'
        self.actionRecord.setIcon(self.icon_rec_idle)

    def send_reponse(self, command_id, response):
        """
        Sends response on received command to outQueue
        """
        msg=win32com.client.Dispatch("MSMQ.MSMQMessage")
        msg.Label=f"{command_id}"
        msg.Body=f"{response}"
        msg.Send(self.outQueue)
        # set active msg_id to None to indicate that all processed messages got responses
        self.active_external_msg_id = None

    def external_command_received(self, res):
        """
        Only accept commands/instructions that are defined in self.inQueue.legal_external_commands. 
        
        RESPONSE STRUCTURE:
        all responses will include: message_id and message_body
        - massage_id is an identifier of the message. It serves for clients to connect responses with messages they send.
        - message_body contains the response content. The content is structured and has three keys: 'cmd_status', err_msg 
          and 'response'. The last two are optional. 
        The meaning of these keys is as follows:
        - 'cmd_status': tells the status of the message. It can be eirther 'True' if command succedded, or False otherwise.
        - 'err_msg': if an error occured (cmd_status=False), the error will be described in this field. Otherwise this field 
           will not be included in the massage_body
        - 'response': if the command sent to the server expects a reponse (e.g. sessionid when start_dict is triggered), this
           response will be included in this field. Otherwise the filed will not be included in the message body.

        The three fields are separated with the pipe symbol "|"  
        """

        self.active_external_msg_id = None
        msg_id = res['label']
        msg_components = res['body'].split("|")
        command = msg_components[0]
        arguments = [msg_components[i] for i in range(1, len(msg_components))]

        # if user not authenticated, ignore everything except login/logout, show/hide toolbar, and get_status commands
        if not self.user_authenticated and msg_components[0].lower() not in ("login", 'status', 'logout', 'show_toolbar', 'hide_toolbar'):
            logger.debug(f"External command received: {res}")
            logger.debug(f"User not authenticated, external command will be ignored...")
            self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: User not authenticated - command ignored.")
            return

        # if in the integration debug mode, show the command
        if _integration_debug_mode:
            self.label_interim.setStyleSheet("color: red")
            self.label_interim.setText(f"Ext: {res['body'][:_interim_display_len]}...")

        if command.lower() not in self.inQueue.legal_external_commands:
            logger.debug(f"Unknown external command received: {command}")
            return
        else:
            logger.debug(f"External request received: label: {msg_id}, body: {res['body']}")
            self.active_external_msg_id = msg_id

            # login
            if command.lower() == 'login':
                try:
                    if not self.user_authenticated:
                        if len(arguments)!=3: 
                            logger.debug(f'Syntax error in external command: {command}\nUsege: login |<username>|<password>|<hostname>')
                            self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: Syntax error. Usage: login |<username>|<password>|<hostname>")
                        else:
                            self.lineEdit_username.setText(arguments[0])
                            self.lineEdit_password.setText(arguments[1])
                            self.lineEdit_host.setText(arguments[2])
                            if self.authenticate():
                                self.send_reponse(msg_id, 'cmd_status=True')
                            else:
                                self.send_reponse(msg_id, 'cmd_status=False | err_msg=Unable to authenticate this user!')
                    else:
                        self.send_reponse(msg_id, 'cmd_status=True')
                except Exception as e:
                    self.send_reponse(msg_id, f'cmd_status=False | err_msg={e}')

            # logout
            if command.lower() == 'logout':
                try:
                    if self.user_authenticated:
                        self.logout()
                    self.send_reponse(msg_id, 'cmd_status=True')
                except Exception as e:
                    self.send_reponse(msg_id, f'cmd_status=False | err_msg={e}')

            # status
            elif command.lower() == 'status':
                # return recording status (idle, recording, pause), buffering status (True, False), Dictaphone status (True, False)
                self.send_reponse(msg_id, 'cmd_status=True | response={authenticated='+f'{self.user_authenticated}, status={self.recording_state}, buffering={self.buffering}, dictaphone_mode={self.actionDictaphone.isChecked()}'+'}')
            
            # start_dict
            elif command.lower() == 'start_dict':
                # only proceed if in idle or pause state and command syntax ok
                if self.recording_state not in ('idle', 'pause'):
                    logger.debug(f"ERROR: dictation can't be started in {self.recording_state} state. External command will be ignored.")
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: can't start recording in {self.recording_state} state")
                elif len(arguments) != 4: 
                    logger.debug(f'Syntax error in external command: {command}\nUsege: start_dict |<window handle>|<patient id>|<session_id>|<notes>\nThe third and fourth arguments can be null, i.e. an empty string, session_id, is optional. Use to continue sesission.')
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: Syntax error. Usege: start_dict |<window handle>|<patient id>|<session_id>|<notes>\nThe third and fourth arguments can be null, i.e. an empty string.")
                else:
                    # Extract session name and sessionid if provided
                    # NOTE: PAUSE mode can only be triggered by pressing toolbar or dictaphone button, or via voice 
                    # command 'pause'. If the listener is in pause mode and sessionid is provided via the external 
                    # command, the provided session will be continued and not the one from which the pause was triggered. 
                    # Note also that when a session is continued, the last provided PID will be used to name the session.
                    self.continue_sessionId = None
                    if arguments[2] != "":
                        if arguments[2].isnumeric():
                            self.continue_sessionId = arguments[2]
                        else:
                            logger.warning(f"Value {arguments[2]} not recognised as a valid session identifier. New session will be created!") 
                    # remember notes if set by the caller
                    if arguments[3] != "":
                        self.session_notes = arguments[3]
                    else:
                        self.session_notes = None 

                    # focus to window handle and start dictation. 
                    if arguments[0].isnumeric():
                        # first show toolbar if hidden in sys tray
                        self.show()
                        self.wtof = int(arguments[0])
                        pid = arguments[1]
                        self.sessionName = pid
                        if self.wtof != windll.user32.GetForegroundWindow():
                            set_focus_to_window_handle(self.wtof)
                            
                        self.record()
                    else:
                        logger.debug(f"Syntax error in external command: {command}\nArgument <window handle> must be convertable to integer.")
                        self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: Syntax error. <window handle> can't be converted to integer!")

            # stop_dict
            # NOTE: when stop_dict is received, stop_recording will be triggered which will in turn reset the buffering state and buffer.
            elif command.lower() == 'stop_dict':
                # only proceed if in recording or pause state and command syntax ok
                if self.recording_state not in ('recording', 'pause'):
                    logger.debug(f"Can't stop dictation - not in recording or pause state. State: {self.recording_state}")
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: can't stop dictation - not in recording or pause state. State: {self.recording_state}")
                    return
                elif len(arguments) != 1: 
                    logger.debug(f'Syntax error in external command: {command}\nUsege: stop_dict |<sesion id>')
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: syntax error in external command. Usege: stop_dict |<sesion id>")
                else:
                    sessionid = arguments[0]
                    if not sessionid.isnumeric() or (int(sessionid) != self.stt.last_sessionId):
                        logger.warning(f"Received command does not refer to the current session: current: {self.stt.last_sessionId} - refered session {sessionid}")
                        self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: received command does not refer to the current session!")
                    else:    
                        # stop recording
                        self.stop_recording()
                        self.recording_state = 'idle'
                        self.send_reponse(msg_id, "cmd_status=True")

            # pause
            # NOTE: this command is deprecated as it can be simulated by simply stopping the current session and then
            # using start_dict with the sessionid to be continued 
            elif command.lower() == 'pause':
                # only eligible if in recording state
                if self.recording_state != 'recording':
                    logger.debug(f"Can't pause dictation - not in recording state. State: {self.recording_state}")
                    return
                elif len(arguments) != 1: 
                    logger.debug(f'Syntax error in external command: {command}\nUsege: pause |<sesion id>')
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: syntax error. Usege: pause |<sesion id>") 
                else:
                    sessionid = arguments[0]
                    if not sessionid.isnumeric() or (int(sessionid) != self.stt.last_sessionId):
                        logger.warning(f"Received command does not refer to the current session: current: {self.stt.last_sessionId} - refered session {sessionid}")
                        self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: received command does not refer to the current session!")
                    else:    
                        # pause recording
                        self.pause_recording()
                        self.send_reponse(msg_id, "cmd_status=True")

            # show toolbar
            elif command.lower() == 'show_toolbar':
                if len(arguments) not in (0, 2) or (len(arguments)==2 and (not arguments[0].isnumeric() or not arguments[1].isnumeric())):
                    logger.debug(f'Syntax error in external command: {command}\nUsege: show_toolbar [|<xpos>|<ypos>]\nThe <xpos> and <ypos> arguments are optional.')
                    self.send_reponse(msg_id, f"cmd_status=False | err_msg=FAILED: Syntax error. Usage: show_toolbar [|<xpos>|<ypos>]. The <xpos> and <ypos> arguments are optional.")
                else:
                    if len(arguments)==2:
                        self.move(int(arguments[0]), int(arguments[1]))
                    self.show()
                    self.send_reponse(msg_id, "cmd_status=True")

            # hide toolbar
            elif command.lower() == 'hide_toolbar':
                self.hide()
                self.send_reponse(msg_id, "cmd_status=True")

    def queue_listener_stopped(self):
        logger.debug("MSMQ listener stopped!")

    def start_recording(self):
        """
        Before starting new session, check settings from gui/settings.yaml
        and reapply if neccessary.

        """
        
        # minimize and start
        self.show_interims()

        # make the destination window active - use window title from the settings 
        # except when window handle is given via an external command and self.wtof holds handle id 
        if self.wtof is None:
            wtof = self.lineEdit_destination.text().strip()
            if wtof:
                set_focus_to_window(wtof)
        else:
            # if external command is triggering the recording, check whether session id is provided
            # in this case make sure to continue the session as if it was in pause mode
            if self.continue_sessionId:
                self.stt.last_sessionId = self.continue_sessionId 

        print("BUFFER: ", self.buffer)
        print("BUFFERING STATE: ", self.buffering)
        print("RECORDING STATE: ", self.recording_state)

        self.streamer = Streamer(
            stt=self.stt,
            settings=self.settings,
            spp_codes=self.spp_codes,
            dictation_type=self.dictation_type,
            continue_session=(self.recording_state=='pause' or (self.continue_sessionId is not None)),
            buffering = self.buffering,
            buffer = self.buffer,
            dictaphone_mode = self.actionDictaphone.isChecked(),
            templates=self.templates,
            )
        self.streamer.signals.result.connect(self.print_output)
        self.streamer.signals.finished.connect(self.thread_complete)
        self.streamer.signals.status.connect(self.display_status)
        self.streamer.signals.error.connect(self.handle_streaming_errors)
        # execute
        self.threadpool.start(self.streamer)

    def toggle_button(self, btn, on:Boolean):
        button = None
        if btn in ('<b>', '</b>'):
            button = self.pushButton_bold
        elif btn in ('<i>', '</i>'):
            button = self.pushButton_italic
        elif btn in ('<u>', '</u>'):
            button = self.pushButton_underline
        elif btn in ('<uc>', '</uc>'):
            button = self.pushButton_upper
        elif btn in ('<lc>', '</lc>'):
            button = self.pushButton_lower
        elif btn in ('<spell>', '</spell>'):
            button = self.pushButton_spell
        if button:
            if not on:
                button.setStyleSheet("background-color : lightgrey; border-style: solid; border: 1px solid grey")
            else:
                button.setStyleSheet("background-color : lightblue; border-style: solid; border: 1px solid blue")

    def select_output(self):
        radioButton = self.sender()
        if radioButton.isChecked():
            if radioButton == self.radioButton_plain:
                self.pushButton_bold.setVisible(False)
                self.pushButton_italic.setVisible(False)
                self.pushButton_underline.setVisible(False)
                self.label_interim.setGeometry(scale(100), scale(5), scale(240), scale(21))
                self.dictation_type="txt"
            else:
                self.pushButton_bold.setVisible(True)
                self.pushButton_italic.setVisible(True)
                self.pushButton_underline.setVisible(True)
                self.label_interim.setGeometry(scale(185), scale(5), scale(240), scale(21))
                if self.radioButton_rtf.isChecked():
                    self.dictation_type="rtf"
                elif self.radioButton_docx.isChecked():
                    self.dictation_type="docx"
                else:
                    self.dictation_type="b21"

    def open_send_logs(self):
        sendlog_dialog = QtWidgets.QDialog()
        sendlog_dialog.ui = SendLog(
            gui_object=sendlog_dialog, 
            logger_folder=self.logger_folder,
            tb_backend=self.stt,
            username=self.lineEdit_username.text(),
            )
        # make window stay on top
        sendlog_dialog.setWindowFlags(Qt.WindowStaysOnTopHint)
        sendlog_dialog.exec()

    def open_commands_helper(self):
        self.commands_helper = QtWidgets.QDialog()
        self.commands_helper.ui = Dictation_commands(self.commands_helper)

        self.COMMANDS_dialog_opened=True
        if not self.isActiveWindow():
            logger.debug(f"Active window when COMMANDS is opening and is without focus: {win32gui.GetWindowText(win32gui.GetForegroundWindow())}")
            wtof = ".*MainWindow.*"
            wtof = "TB-listener: MainWindow"
            if wtof:
                set_focus_to_window(wtof)

        if self.commands_helper.exec()==QDialog.Rejected:
            # tell the streamer that window was closed
            if self.streamer:
                self.streamer.helper_window_closed()
            # set focus back to the destination window
            self.COMMANDS_dialog_opened=False
            wtof = self.lineEdit_destination.text().strip()
            if wtof:
                set_focus_to_window(wtof)

    def open_add_word_dialog(self):
        self.add_word_dialog = QtWidgets.QDialog()
        self.add_word_dialog.ui = Add_Word_Dialog(
            gui_object=self.add_word_dialog,
            truebar_object=self.stt,
            tb_config=self.tb_current_config,
            )
        self.add_word_dialog.setWindowFlags(Qt.WindowStaysOnTopHint)
        self.add_word_dialog.exec()

    def open_dialog(self, whndl=None, pid=None):
        # if in recording mode, put in pause
        if self.recording_state == 'recording':
            self.pause_recording()
        # if in pause mode
        elif self.recording_state=='pause':
            # set self.wtof and self.continue_sessionId to None to indicate that the command is not comming from an external command
            self.wtof = None
            self.continue_sessionId = None
            self.record()
        # if idle
        else:
            # if the user switched off this feature in the settings, ignore it and just start dictation
            # self.continue_sessionId refers to session start initiated via an external command and is None in all cases here
            logger.debug(f"show-dialog-PID: {get_settings_value(self.settings, ['general', 'show-dialog-PID'])}")
            if not get_settings_value(self.settings, ['general', 'show-dialog-PID']):
                self.continue_sessionId = None
                self.record()
                return

            dialog = QtWidgets.QDialog()
            dialog.ui = Ui_Dialog_pid()
            dialog.ui.setupUi(dialog, self.settings, whndl=whndl, pid=pid)

            # make window stay on top
            dialog.setWindowFlags(Qt.WindowStaysOnTopHint)
            
            # Show/hide notification
            """
            if self.qt_settings.value('suppress-notifications')=='true':
                dialog.ui.textBrowser_notification.setVisible(False)
                dialog.ui.checkBox_dontshow.setVisible(False)
                dialog.resize(300, 110)
            else:
                dialog.ui.textBrowser_notification.setVisible(True)
                dialog.ui.checkBox_dontshow.setVisible(True)
                dialog.resize(300, 220)
            """

            # position dialog right/left from the toolbar
            if self.pos().x() + self.size().width() + scale(10) + dialog.frameGeometry().width() > QDesktopWidget().availableGeometry().width():
                dialog.move(
                    self.pos().x() - dialog.frameGeometry().width() - scale(10), 
                    self.pos().y()
                    )    
            else:
                dialog.move(
                    self.pos().x() + self.frameGeometry().width() + scale(10), 
                    self.pos().y()
                    )
            
            # open dialog
            self.PID_dialog_opened=True
            
            if not self.isActiveWindow():
                logger.debug(f"Active window when PID is opening and is without focus: {win32gui.GetWindowText(win32gui.GetForegroundWindow())}")
                # make the destination window active
                wtof = get_settings_value(self.settings, ['localization', 'main-window-title'])
                #wtof = "TB-listener"
                if wtof:
                    set_focus_to_window(wtof)

            if dialog.exec_()==QDialog.Accepted:
                # construct session name as required in settings
                session_template = get_settings_value(self.settings, ['localization', 'session-name-template'])
                if session_template:
                    self.sessionName = session_template.format(
                        label1=dialog.ui.lineEdit_pid.text(),
                        label2=dialog.ui.lineEdit_ido.text()
                        )
                else:
                    self.sessionName = dialog.ui.lineEdit_pid.text()
                logger.info(f"Starting dictation for the patient with id {self.sessionName}")
                # setting self.wtof to None to indicate that the command is not comming from an external command
                self.wtof = None
                self.continue_sessionId = None
                self.record()
            self.PID_dialog_opened=False

    def open_detailed_settings(self):
        detailed_settings_dialog = QtWidgets.QDialog()
        detailed_settings_dialog.ui = DetailedSettings(
            gui_object=detailed_settings_dialog,
            settings=self.settings,
            truebar_object=self.stt,
            )
        detailed_settings_dialog.setWindowFlags(Qt.WindowStaysOnTopHint)
        detailed_settings_dialog.exec()

class Ui_Dialog_pid(object):
    def setupUi(self, Dialog_pid, settings, whndl=None, pid=None):
        # set Qsettings
        self.qt_settings = QSettings('Vitasis', 'TB-listener')
        self.settings = settings
        #
        Dialog_pid.setObjectName("Dialog_pid")
        # make window fixed size
        Dialog_pid.setFixedSize(int(303*scalingFactor), int(110*scalingFactor))
        #Dialog_pid.resize(scale(303), scale(110))

        font = QtGui.QFont()
        font.setBold(True)
        font.setWeight(75)
        
        self.label_pid = QtWidgets.QLabel(Dialog_pid)
        self.label_pid.setGeometry(QtCore.QRect(scale(20), scale(22), scale(91), scale(16)))
        self.label_pid.setObjectName("label_pid")
        self.label_pid.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)

        self.lineEdit_pid = QtWidgets.QLineEdit(Dialog_pid)
        self.lineEdit_pid.setGeometry(QtCore.QRect(scale(120), scale(20), scale(150), scale(20)))
        self.lineEdit_pid.setFont(font)
        self.lineEdit_pid.setObjectName("lineEdit_pid")
        pid_default = get_settings_value(self.settings, ['localization', 'session-name-label1-default'])
        if pid_default:
            self.lineEdit_pid.setText(pid_default)

        self.label_ido = QtWidgets.QLabel(Dialog_pid)
        self.label_ido.setGeometry(QtCore.QRect(scale(20), scale(47), scale(91), scale(16)))
        self.label_ido.setObjectName("label_ido")
        self.label_ido.setAlignment(QtCore.Qt.AlignRight|QtCore.Qt.AlignTrailing|QtCore.Qt.AlignVCenter)
        
        self.lineEdit_ido = QtWidgets.QLineEdit(Dialog_pid)
        self.lineEdit_ido.setGeometry(QtCore.QRect(scale(120), scale(45), scale(150), scale(20)))
        self.lineEdit_ido.setFont(font)
        self.lineEdit_ido.setObjectName("lineEdit_ido")
        ido_default = get_settings_value(self.settings, ['localization', 'session-name-label2-default'])
        if ido_default:
            self.lineEdit_ido.setText(ido_default)

        self.pushButton_cancel = QtWidgets.QPushButton(Dialog_pid)
        self.pushButton_cancel.setGeometry(QtCore.QRect(scale(200), scale(70), scale(71), scale(23)))
        self.pushButton_cancel.setDefault(False)
        self.pushButton_cancel.setObjectName("pushButton_cancel")
        
        self.pushButton_record = QtWidgets.QPushButton(Dialog_pid)
        self.pushButton_record.setGeometry(QtCore.QRect(scale(120), scale(70), scale(71), scale(23)))
        self.pushButton_record.setDefault(True)
        self.pushButton_record.setObjectName("pushButton_record")
        
        self.textBrowser_notification = QtWidgets.QTextBrowser(Dialog_pid)
        self.textBrowser_notification.setGeometry(QtCore.QRect(scale(30), scale(110), scale(241), scale(91)))
        self.textBrowser_notification.setStyleSheet("border-style: solid;\n"
"background-color: rgb(scale(231), scale(231), scale(231));\n"
"border-width: 1px;\n"
"border-color: rgb(scale(206), scale(206), scale(206));")
        self.textBrowser_notification.setObjectName("textBrowser_notification")
        self.textBrowser_notification.setVisible(False)
        self.checkBox_dontshow = QtWidgets.QCheckBox(Dialog_pid)
        self.checkBox_dontshow.setGeometry(QtCore.QRect(scale(40), scale(160), scale(191), scale(17)))
        self.checkBox_dontshow.setObjectName("checkBox_dontshow")
        self.checkBox_dontshow.stateChanged.connect(self.change_notifications)
        self.checkBox_dontshow.setVisible(False)

        self.retranslateUi(Dialog_pid)

        # if pid is received over external request, preload it to lineEdit_pi
        if pid:
            self.lineEdit_pid.setText(pid)

        # if 'Prekini' is clicked, send reject signal and close dialog - do nothing further
        self.pushButton_cancel.clicked.connect(Dialog_pid.reject)
        # if 'Začni' is clicked, first validate if PID was entered ...
        self.pushButton_record.clicked.connect(lambda: self.validate_and_start(Dialog_pid))
        # the next ine is confusing: check https://stackoverflow.com/questions/2462401/problem-in-understanding-connectslotsbyname-in-pyqt
        QtCore.QMetaObject.connectSlotsByName(Dialog_pid)
        self.button_pressed = None

    def retranslateUi(self, Dialog_pid):
        _translate = QtCore.QCoreApplication.translate
        Dialog_pid.setWindowTitle(_translate("Dialog_pid", "Oznaka zapisa"))
        self.label_pid.setText(_translate("Dialog_pid", _nvl(get_settings_value(self.settings, ['localization', 'session-name-label1']), "Oznaka pacienta")))
        self.label_ido.setText(_translate("Dialog_pid", _nvl(get_settings_value(self.settings, ['localization', 'session-name-label2']), "Številka obravnave")))
        self.pushButton_cancel.setText(_translate("Dialog_pid", "Prekini"))
        self.pushButton_record.setText(_translate("Dialog_pid", "Začni"))
        self.textBrowser_notification.setHtml(_translate("Dialog_pid", "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\" \"http://www.w3.org/TR/REC-html40/strict.dtd\">\n"
"<html><head><meta name=\"qrichtext\" content=\"1\" /><style type=\"text/css\">\n"
"p, li { white-space: pre-wrap; }\n"
"</style></head><body style=\" font-family:\'MS Shell Dlg 2\'; font-size:8.25pt; font-weight:400; font-style:normal;\">\n"
"<p style=\" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;\">Po pritisku na gumb <span style=\" font-weight:600;\">Začni</span> najprej preverite, če je kurzor v oknu, v katerega želite narekovati besedilo. Šele nato začnite z narekovanjem! </p></body></html>"))
        self.checkBox_dontshow.setText(_translate("Dialog_pid", "Ne prikazuj več tega obvestila"))

    def validate_and_start(self, Dialog_pid):
        if self.lineEdit_pid.text().strip() != "":
            Dialog_pid.accept()
        else:
            self.lineEdit_pid.setStyleSheet("border: 1px solid red;")
            
    def change_notifications(self):
        # read and change settings
        if self.checkBox_dontshow.checkState()==0:
            self.qt_settings.setValue('suppress-notifications', False)
        else:
            self.qt_settings.setValue('suppress-notifications', True)

class Dictation_commands(QtWidgets.QDialog, Ui_Dialog_commands):
    def __init__(self, gui_object):
        super().__init__()
        # render gui
        self.setupUi(gui_object)
        # make window fixed size
        gui_object.setFixedSize(int(591*scalingFactor), int(693*scalingFactor))
        # make window stay on top
        self.setWindowFlags(
            self.windowFlags() |  
            QtCore.Qt.WindowStaysOnTopHint
        )
        # connect pushbutton click to action close
        self.pushButton_close.clicked.connect(gui_object.close)

class Add_Word_Dialog(QtWidgets.QDialog, Ui_Dialog_add_word):
    def __init__(self, gui_object, truebar_object, tb_config):
        super().__init__()
        # to work with the BE, Truebar object, created within the mainwindow is used
        self.stt = truebar_object
        self.tb_current_config = tb_config
        # render gui
        self.setupUi(gui_object)
        # make window fixed size 437|512
        gui_object.setFixedSize(int(242*scalingFactor), int(439*scalingFactor))
        # make window stay on top
        self.setWindowFlags(
            self.windowFlags() |  
            QtCore.Qt.WindowStaysOnTopHint
        )
        # load icons
        icon_download = QtGui.QIcon()
        icon_download.addPixmap(QtGui.QPixmap(os.path.join(basedir, "download.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.pushButton_readDict.setIcon(icon_download)
        self.icon_collapse = QtGui.QIcon()
        self.icon_collapse.addPixmap(QtGui.QPixmap(os.path.join(basedir, "collapse.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        self.icon_expand = QtGui.QIcon()
        self.icon_expand.addPixmap(QtGui.QPixmap(os.path.join(basedir, "more.png")), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        # initially the detailed bar is collapsed
        self.pushButton_expand.setIcon(self.icon_expand)
        self.dialog_expanded = False
        
        # configure badge
        self.label_badge.setStyleSheet("color: white; background-color: blue; border-radius: 12px;")

        # connect pushbuttons
        self.pushButton_add_pron.clicked.connect(lambda: self.add_pron())
        self.pushButton_del_pron.clicked.connect(lambda: self.del_pron())
        self.pushButton_update_pron.clicked.connect(lambda: self.update_pron())
        self.pushButton_dict_close.clicked.connect(gui_object.close)
        self.pushButton_dict_save.clicked.connect(lambda: self.save_dict())
        self.pushButton_readDict.clicked.connect(lambda: self.read_dictionary())
        self.pushButton_expand.clicked.connect(lambda: self.show_details(gui_object))
        self.pushButton_update_model.clicked.connect(lambda: self.start_model_update())

        # connect listWidget_pron
        self.listWidget_pron.currentRowChanged.connect(lambda: self.change_current_pron())

        # get frequency classes
        [{"id":1,"label":"zelo redko"},{"id":2,"label":"redko"},{"id":3,"label":"občasno"},{"id":4,"label":"pogosto"},{"id":5,"label":"zelo pogosto"}]
        frequency_classes = self.stt.get_frequency_classes()
        if frequency_classes:
            self.comboBox_tf.addItems([None]+[fc['label'] for fc in frequency_classes])
        else:
            logger.warning(f"Couldn't retrieve frequency classes from the server!")
        
        # init fields
        self.init_fields()

        # set validation for pronunciations - only allowed phonemes are acceptable
        self.phoneme_validator=None
        res=self.get_phoneset()
        if res:
            allowed_phonemes = "|" + "".join(res['nonsilentPhones'].split(" "))
            reg_ex = QRegExp(rf"[{allowed_phonemes}]+")
            self.phoneme_validator = QRegExpValidator(reg_ex, self.lineEdit_curr_pron)
            self.lineEdit_curr_pron.setValidator(self.phoneme_validator)

        # read data for badge on new word 
        self.read_word_stat()

        # set variable for timer
        self.timer = None
        
        # read data about model update
        self.qt_settings = QSettings('Vitasis', 'TB-listener')
        self.model_updates = self.qt_settings.value('model-updates') or dict()
        print(self.model_updates)

    def init_fields(self):
        self.comboBox_tf.setCurrentIndex(0)
        self.lineEdit_curr_pron.clear()
        self.lineEdit_word.clear()
        self.listWidget_pron.clear()
        self.label_word_status.clear()
        self.label_action_status.clear()
        self.label_update_status.clear()
        self.lineEdit_word.setFocus()

    def get_phoneset(self):
        return self.stt.get_model_phoneset(
            language = self.tb_current_config["stt"]["language"]["value"], 
            domain = self.tb_current_config["stt"]["domain"]["value"],
            model = self.tb_current_config["stt"]["model"]["value"], 
        )
        
    def read_dictionary(self):
        self.lineEdit_curr_pron.clear()
        self.listWidget_pron.clear()
        self.comboBox_tf.setCurrentIndex(0)
        self.label_action_status.clear()

        res = self.stt.read_dictionary(
            language = self.tb_current_config["stt"]["language"]["value"], 
            domain = self.tb_current_config["stt"]["domain"]["value"],
            model = self.tb_current_config["stt"]["model"]["value"],
            token=self.lineEdit_word.text()
        )
        
        if res:
            print(res)
            # if multiple records exist for the same token, log this
            if len(res)>1:
                logger.warning(f"Multiple records received for the word {self.lineEdit_word.text()}.")
            
            if 'status' in res[0]:
                if res[0]["status"] == 'NEW':
                    self.label_word_status.setText('nova beseda')
                    self.label_word_status.setStyleSheet("color: darkred")
                elif res[0]["status"] == 'IN_DICTIONARY': 
                    self.label_word_status.setText('beseda je že v slovarju izbranega modela')
                    self.label_word_status.setStyleSheet("color: green")
                else: 
                    self.label_word_status.setText('beseda se trenutno dodaja v slovar izbranega modela.')
                    self.label_word_status.setStyleSheet("color: green")
                
                if "pronunciations" in res[0]:
                    self.listWidget_pron.addItems([el["text"] for el in res[0]['pronunciations'] if el["saved"]])
                    # select first pron
                    self.listWidget_pron.setCurrentRow(0)
                    if self.listWidget_pron.currentItem():
                        self.lineEdit_curr_pron.setText(self.listWidget_pron.currentItem().text())

                if res[0]["status"] != 'NEW':
                    if "frequencyClassId" in res[0]:
                        self.comboBox_tf.setCurrentIndex(res[0]["frequencyClassId"])
                else:
                    self.comboBox_tf.setCurrentIndex(3)
                    
        self.lineEdit_word.setFocus()

    def add_pron(self):
        if self.lineEdit_curr_pron.text() not in [self.listWidget_pron.item(i).text() for i in range(self.listWidget_pron.count())]:
            self.listWidget_pron.addItem(self.lineEdit_curr_pron.text())
    
    def del_pron(self):
        if self.listWidget_pron.currentRow() != -1:
            self.listWidget_pron.takeItem(self.listWidget_pron.currentRow())
            self.lineEdit_curr_pron.clear()
        if self.listWidget_pron.count()>0:
            self.listWidget_pron.setCurrentRow(0)
            self.lineEdit_curr_pron.setText(self.listWidget_pron.currentItem().text())
            self.pron_current_row=0
        else:
            self.pron_current_row=None

    def update_pron(self):
        if self.pron_current_row:
            self.listWidget_pron.setCurrentRow(self.pron_current_row)
            self.listWidget_pron.currentItem().setText(self.lineEdit_curr_pron.text())

    def save_dict(self):
        if self.listWidget_pron.count() > 0:
            word_entry = dict(
                text=self.lineEdit_word.text(),
                frequencyClassId=self.comboBox_tf.currentIndex(),
                pronunciations=[dict(text=self.listWidget_pron.item(i).text()) for i in range(self.listWidget_pron.count())]
            )
            if self.stt.save_dictionary(
                language = self.tb_current_config["stt"]["language"]["value"], 
                domain = self.tb_current_config["stt"]["domain"]["value"],
                model = self.tb_current_config["stt"]["model"]["value"],
                word_entry=word_entry,
                ):
                self.init_fields()
                self.label_action_status.setText("Beseda uspešno shranjena.")
                self.label_action_status.setStyleSheet("color: green")
                
            else:
                self.label_action_status.setText("Pri shranjevanju je prišlo do napake!")
                self.label_action_status.setStyleSheet("color: red")
                logger.error(f"Error when saving word entry {word_entry} to dictionary!")
                
        else:
            self.label_action_status.setText("Beseda nima nobene izgovorjave!")
            self.label_action_status.setStyleSheet("color: red")

    def change_current_pron(self):
        if self.listWidget_pron.currentItem():
            self.pron_current_row = self.listWidget_pron.currentRow()
            self.lineEdit_curr_pron.setText(self.listWidget_pron.currentItem().text())
            if self.phoneme_validator:
                if self.phoneme_validator.validate(self.lineEdit_curr_pron.text(), 0)[0] != QtGui.QValidator.Acceptable:
                    self.lineEdit_curr_pron.setStyleSheet("border-style: solid; border-width: 1px; border-color: red")
                else:
                    self.lineEdit_curr_pron.setStyleSheet("border-style: solid; border-width: 1px; border-color: grey")

    def read_word_stat(self):
        res = self.stt.get_model_word_count(
            language = self.tb_current_config["stt"]["language"]["value"], 
            domain = self.tb_current_config["stt"]["domain"]["value"],
            model = self.tb_current_config["stt"]["model"]["value"]
            )
        if res:
            self.label_badge.setText(str(res.get('wordsNew', None)))
            if res.get('wordsNew', 0) > 0:
                self.label_badge.setVisible(True)
            else:
                self.label_badge.setVisible(False) 
            return True
        else:
            self.label_badge.setVisible(False)
            return False

    def show_details(self, gui_object):
        if not self.dialog_expanded:
            self.refresh_model_update_status()
            # show badge if new words waiting for the update
            self.read_word_stat()
            gui_object.setFixedHeight(int(_dictionary_frame_height_L*scalingFactor))
            self.pushButton_expand.setIcon(self.icon_collapse)
            # start timer to check updater status every 3 sec - timer will stop if status is FAILED or NOT_RUNNING
            self.timer = QTimer()
            self.timer.setInterval(3000)
            self.timer.timeout.connect(self.refresh_model_update_status)
            self.timer.start()
        else:
            gui_object.setFixedHeight(int(_dictionary_frame_height_S*scalingFactor))
            self.pushButton_expand.setIcon(self.icon_expand)
            # stop timer
            if self.timer:
                self.timer.stop()
        self.dialog_expanded = not self.dialog_expanded

    def start_model_update(self):
        language = self.tb_current_config["stt"]["language"]["value"]
        domain = self.tb_current_config["stt"]["domain"]["value"]
        model = self.tb_current_config["stt"]["model"]["value"]
        
        if show_message_box(
            message='Posodobitev modela lahko traja več kot 20 minut. Po končeni posodobitvi bodo vse seje z obstoječim modelom samodejno prekinjene! Nov model bo na voljo kot nova izbira v okviru nastavitev. Želite nadaljevati?',
            box_title='Opozorilo',
            button_yes_text='Nadaljuj',
            button_no_text='Prekliči',
            box_type='Warning',
            num_of_buttons=2
            ) == 1:
            # start model updater
            if self.stt.start_model_update(language=language, domain=domain, model=model):
                logger.debug("Model update started!")
                self.label_update_status.setText("Model se prenavlja ...")
                self.label_update_status.setStyleSheet(f"color: {cl_message_success}")
                # save starting time
                self.model_updates[f"{language}{domain}{model}"].update(start_time=time.time())
                self.qt_settings.setValue('model-updates', self.model_updates)
                # start timer
                if not self.timer.isActive():
                    self.timer.start()
            else:
                logger.error("Model update didn't start!") 
        else:
            print(self.stt.check_model_update_status(
                language = language, 
                domain = domain,
                model = model,
            ))
        
    def refresh_model_update_status(self):
        language = self.tb_current_config["stt"]["language"]["value"]
        domain = self.tb_current_config["stt"]["domain"]["value"]
        model = self.tb_current_config["stt"]["model"]["value"]
        
        self.label_update_status.setText("")
        
        model_updater_status = self.stt.check_model_update_status(language=language, domain=domain, model=model)
        if not model_updater_status:
            self.label_update_status.setText("Servis za prenovo trenutno ni na voljo!")
            self.label_update_status.setStyleSheet(f"color: {cl_message_failed}")
            return

        # if servis is not running and no data in regisry yet, set empty dict and 
        # then update. There will be no start_time key until model updater is started  
        if f"{language}{domain}{model}" not in self.model_updates:
            self.model_updates[f"{language}{domain}{model}"]=dict()
        self.model_updates[f"{language}{domain}{model}"].update(
            status=model_updater_status.get('statusCode', 'UNKNOWN'),
            last_check=time.strftime('%H:%M:%S', time.gmtime(time.time()))
            )
        # refresh registry entry
        self.qt_settings.setValue('model-updates', self.model_updates)
        
        if model_updater_status and model_updater_status.get('enableModelUpdating', False):
            logger.debug(f"Model update status: {model_updater_status}")
            if model_updater_status['statusCode']=="FAILED":
                # if last attempt failed, show message and allow restart. Stop timer if running.
                self.label_update_status.setText("Zadnji poskus prenove neuspešen!")
                self.label_update_status.setStyleSheet(f"color: {cl_message_failed}")
                self.pushButton_update_model.setEnabled(True)
                if self.timer:
                    self.timer.stop()
            elif model_updater_status['statusCode']=="RUNNING":
                # if running, keep updating status and disable pushButton
                elapsed_time = time.time() - self.model_updates[f"{language}{domain}{model}"].get('start_time', time.time())
                self.label_update_status.setText(f"Model se prenavlja ({time.strftime('%H:%M:%S', time.gmtime(elapsed_time))})")
                self.label_update_status.setStyleSheet(f"color: {cl_message_success}")
                self.pushButton_update_model.setEnabled(False)
            elif model_updater_status['statusCode']=="NOT_RUNNING":
                # if not running, clear status message and enable pushButton if new words > 0
                # stop timer if running
                self.label_update_status.setText("")
                if self.label_badge.text()!="0":
                    self.pushButton_update_model.setEnabled(True)
                else:
                    self.pushButton_update_model.setEnabled(False)
                if self.timer:
                    self.timer.stop()
        else:
            self.pushButton_update_model.setEnabled(False)
        
class SendLog(QtWidgets.QDialog, Ui_Dialog_sendLog):
    def __init__(self, gui_object, logger_folder, tb_backend, username):
        super().__init__()
        # render gui
        self.setupUi(gui_object)
        # make window fixed size
        gui_object.setFixedSize(int(331*scalingFactor), int(137*scalingFactor))
        # make window stay on top
        self.setWindowFlags(
            self.windowFlags() |  
            QtCore.Qt.WindowStaysOnTopHint
        )
        self.logger_folder = logger_folder
        self.tb_backend = tb_backend
        self.username = username
        # connect pushbuttons 
        self.pushButton_confirm.clicked.connect(self.sendlogs)
        self.pushButton_cancel.clicked.connect(gui_object.close)
    
    def sendlogs(self):
        """
        Logs are sent to minio over BE. User must be authenticated!
        """
        self.label_upload_result.setText("Nalagam ...")
        self.label_upload_result.setStyleSheet("color: black")
        # process events to make label text change immediately
        QApplication.processEvents()

        res = self.tb_backend.upload_logs(
            log_folder=self.logger_folder, 
            username=self.username,
            )
       
        if 'all_log_files' in res and 'failed' in res:
            self.label_upload_result.setText(f"<b>{res['all_log_files']-res['failed']}</b> od <b>{res['all_log_files']}</b> datotek uspešno naloženih.")
            self.label_upload_result.setStyleSheet("color: green") 
        else:
            self.label_upload_result.setText("Pri nalaganju log datotek je prišlo do napake!")
            self.label_upload_result.setStyleSheet(f"color: {cl_message_failed}")

class DetailedSettings(QtWidgets.QDialog, Ui_Dialog_detailed_settings):
    def __init__(self, gui_object, settings, truebar_object=None):
        super().__init__()

        logger.debug(f"1. init started")
        QtWidgets.QApplication.instance().focusChanged.connect(self.on_focusChanged)
        self.gui_object = gui_object

        self.settings = settings
        # the class variable settings will change. To keep track on changes, another variable is used, settings_orig
        self.settings_orig = copy.deepcopy(self.settings)
        # to work with teh BE, Truebar object, created within the mainwindow is used
        self.stt = truebar_object
        self.destination_formats = {0:'txt', 1:'rtf', 2:'docx', 3:'b21'}
        self.commands = { 
            '<nl>': 'nova vrsta', 
            '<np>': 'nov odstavek', 
            '<uc>': 'vključi velike črke', 
            '</uc>': 'izključi velike črke', 
            '<lc>': 'vključi male črke', 
            '</lc>': 'izključi male črke',  
            '<b>': 'vključi odebeljeno', 
            '<i>': 'vključi poševno', 
            '<u>': 'vključi podčrtano', 
            '</b>': 'izključi odebeljeno', 
            '</i>': 'izključi poševno', 
            '</u>': 'izključi podčrtano',
            '<cc>': 'začni z veliko začetnico',
            '<alignc>': 'poravnaj na sredino', 
            '<alignl>': 'poravnaj na levo', 
            '<alignr>': 'poravnaj na desno', 
            '<delw>': 'briši besedo', 
            '<dels>': 'briši stavek',
            '<selectall>': 'izberi vse',
            '<copy>': 'zapomni', 
            '<paste>': 'prilepi zapomnjeno besedilo',
            '<pause>': 'pavza', 
            '<end>': 'konec diktata',
            '<ins>': 'vstavi',
            '<showcmd>': 'pokaži ukaze', 
            '<close>': 'zapri', 
            '<space>': 'presledek', 
            '<ok>': 'potrdi',  
            '<next>': 'naprej', 
            '<canc>': 'prekliči', 
            '<spell>': 'črkuj', 
            '</spell>': 'izključi črkovanje', 
        }
        self.commands_help = { 
            'nova vrsta': 'Oznaka: <nl>\n\nPrivzeta akcija: skoči v novo vrsto.', 
            'nov odstavek': 'Oznaka: <np>\n\nPrivzeta akcija: naredi odstavek.', 
            'vključi velike črke': 'Oznaka: <uc>\n\nPrivzeta akcija: vključi velike črke. Velike črke bodo vključene do ukaza "izključi velike črke".', 
            'izključi velike črke': 'Oznaka </uc>\n\nPrivzeta akcija: izključi velike črke.', 
            'vključi male črke': 'Oznaka: <lc>\n\nPrivzeta akcija: vključi male črke. Male črke bodo vključene do ukaza "izključi male črke".', 
            'izključi male črke': 'Oznaka: </lc>\n\nPrivzeta akcija: izključi male črke.',
            'vključi odebeljeno': 'Oznaka: <b>\n\nPrivzeta akcija: vključi odebleljeno pisavo. Odebeljena pisava bo vključena do ukaza "izključi odebeljeno".', 
            'vključi poševno': 'Oznaka: <i>\n\nPrivzeta akcija: vključi poševno pisavo. Poševna pisava bo vključena do ukaza "izključi poševno".', 
            'vključi podčrtano': 'Oznaka: <u>\n\nPrivzeta akcija: vključi podčrtano pisavo. Podčrtana pisava bo vključena do ukaza "izključi podčrtano".', 
            'izključi odebeljeno': 'Oznaka: </b>\n\nPrivzeta akcija: izključi odebeljeno pisavo.', 
            'izključi poševno': 'Oznaka: </i>\n\nPrivzeta akcija: izključi poševno pisavo.', 
            'izključi podčrtano': 'Oznaka: </u>\n\nPrivzeta akcija: izključi podčrtano pisavo.',
            'začni z veliko začetnico': 'Oznaka: <cc>\n\nPrivzeta akcija: narekovano besedilo začni z veliko začetnico.',
            'poravnaj na sredino': 'Oznaka: <alignc>\n\nPrivzeta akcija: besedilo trenutne vrstice/odstavka poravnaj na sredino.', 
            'poravnaj na levo': 'Oznaka: <alignl>\n\nPrivzeta akcija: besedilo trenutne vrstice/odstavka poravnaj na levo.', 
            'poravnaj na desno': 'Oznaka: <alignr>\n\nPrivzeta akcija: besedilo trenutne vrstice/odstavka poravnaj an desno.', 
            'briši besedo': 'Oznaka: <delw>\n\nPrivzeta akcija: briši zadnji znak. Če ukazu briši sledi število n, bo brisanih n znakov.', 
            'briši stavek': 'Oznaka: <dels>\n\nPrivzeta akcija: briši stavek. Ukaz ni podprt.',
            'izberi vse': 'Oznaka: <selectall>\n\nPrivzeta akcija: izberi vse besedilo v dokumentu.',
            'zapomni': 'Oznaka: <copy>\n\nPrivzeta akcija: zapomni si besedilo - ne izpisuj na ekran, temveč hrani v spominu.', 
            'prilepi zapomnjeno besedilo': 'Oznaka: <paste>\n\nPrivzeta akcija: prilepi zapomnjeno besedilo na pozicijo, kjer stoji kurzor.',
            'pavza': 'Oznaka: <pause>\n\nPrivzeta akcija: prekini narek, vendar ne zaključi. Narek bo možno nadaljevati.', 
            'konec diktata': 'Oznaka: <end>\n\nPrivzeta akcija: končaj oziroma zaključi narek.',
            'vstavi': 'Oznaka: <ins>\n\nPrivzeta akcija: ukaz nima privzete akcije.',
            'pokaži ukaze': 'Oznaka: <showcmd>\n\nPrivzeta akcija: pokaži vse možne ukaze.', 
            'zapri': 'Oznaka: <close>\n\nPrivzeta akcija: zapri okno - ukaz bo deloval, če je odprto okno s seznamom ukazov.', 
            'presledek': 'Oznaka: <space>\n\nPrivzeta akcija: naredi presledek.', 
            'potrdi': 'Oznaka: <ok>\n\nPrivzeta akcija: ukaz nima privzete akcije.',  
            'naprej': 'Oznaka: <next>\n\nPrivzeta akcija: ukaz nima privzete akcije.', 
            'prekliči': 'Oznaka: <canc>\n\nPrivzeta akcija: prekliči zadnjo spremembo', 
            'črkuj': 'Oznaka: <spell>\n\nPrivzeta akcija: vključi črkovanje. Pri vključenem črkovanjo se bodo izpisovale le prve črke vsake izgovorjene besede. Npr. če z nekaj premora izgovorite "avto" "hiša" "slovenija", se bo na ekran izpisalo "AHS". Črkovanje bo vključeno do ukaza "izključi črkovanje".', 
            'izključi črkovanje': 'Oznaka: </spell>\n\nPrivzeta akcija: izključi črkovanje.', 
        }
        # initial state of changes
        self.settings_changed = False
        
        # render gui
        self.setupUi(gui_object)

        logger.debug(f"2. GUI rendered")

        # make first tab active
        self.tabWidget.setCurrentIndex(0)
        # clear status label
        self.label_result_status.clear()
        # make window fixed size
        gui_object.setFixedSize(int(819*scalingFactor), int(629*scalingFactor))
        # make window stay on top
        self.setWindowFlags(
            self.windowFlags() |  
            QtCore.Qt.WindowStaysOnTopHint
        )
        # connect pushbuttons
        #self.pushButton_close.clicked.connect(gui_object.close)
        self.pushButton_close.clicked.connect(lambda: self.close_window())
        self.pushButton_save.clicked.connect(lambda: self.save_settings())
        self.pushButton_add_basic_repl.clicked.connect(lambda: self.add_replacement_to_list())
        self.pushButton_del_basic_repl.clicked.connect(lambda: self.delete_replacement_from_list())
        self.pushButton_update_basic_repl.clicked.connect(lambda: self.update_replacement_on_list())
        # connect checkBoxes
        self.checkBox_use_proxy.clicked.connect(lambda: self.init_fields(fields='proxy'))
        self.checkBox_focus_by_key.clicked.connect(lambda: self.init_fields(fields='key'))
        self.checkBox_focus_by_title.clicked.connect(lambda: self.init_fields(fields='title'))
        self.checkBox_listen_external.clicked.connect(lambda: self.init_fields(fields='listener'))
        self.checkBox_numbers.stateChanged.connect(lambda: self.save_changes(fields='numbers'))
        self.checkBox_spp.stateChanged.connect(lambda: self.save_changes(fields='spp'))
        self.checkBox_capitalization.stateChanged.connect(lambda: self.save_changes(fields='capitalization'))
        self.checkBox_beep_on_final.stateChanged.connect(lambda: self.save_changes(fields='beep'))
        self.checkBox_format_notes.stateChanged.connect(lambda: self.save_changes(fields='notes'))
        self.checkBox_show_PID.stateChanged.connect(lambda: self.save_changes(fields='pid'))
        self.checkBox_use_proxy.stateChanged.connect(lambda: self.save_changes(fields='use-proxy'))
        self.checkBox_focus_by_key.stateChanged.connect(lambda: self.save_changes(fields='key'))
        self.checkBox_focus_by_title.stateChanged.connect(lambda: self.save_changes(fields='title'))
        self.checkBox_listen_external.stateChanged.connect(lambda: self.save_changes(fields='listener'))
        self.checkBox_purge_queue.stateChanged.connect(lambda: self.save_changes(fields='purge-queue'))
        self.checkBox_dictaphone_toggle.stateChanged.connect(lambda: self.save_changes(fields='dictaphone'))
        self.checkBox_minimize.stateChanged.connect(lambda: self.save_changes(fields='systray'))
        # connect comboBoxes
        self.comboBox_protocol.currentTextChanged.connect(lambda: self.save_changes(fields='proxy-protocol'))
        # connect other components of the replacements tab
        self.listWidget_basic_repl.currentRowChanged.connect(lambda: self.populate_list_of_replacements())
        self.lineEdit_filter.textChanged.connect(self.filter_replacements)
        # populate fields with settings
        self.checkBox_numbers.setChecked(get_settings_value(self.settings, ['general', 'do-postprocessing', 'numbers']))
        self.checkBox_spp.setChecked(get_settings_value(self.settings, ['general', 'do-postprocessing', 'spp-codes']))
        self.checkBox_capitalization.setChecked(get_settings_value(self.settings, ['general', 'do-postprocessing', 'capitalize-sentences']))
        self.checkBox_beep_on_final.setChecked(get_settings_value(self.settings, ['general', 'beep-on-finals']))
        self.checkBox_format_notes.setChecked(get_settings_value(self.settings, ['general', 'do-postprocessing', 'format-notes']))
        self.checkBox_show_PID.setChecked(get_settings_value(self.settings, ['general', 'show-dialog-PID']))
        self.checkBox_minimize.setChecked(get_settings_value(self.settings, ['general', 'system-tray', 'minimize-after-login']))
        self.checkBox_listen_external.setChecked(get_settings_value(self.settings, ['integration', 'listen-for-external-commands']))
        self.checkBox_purge_queue.setChecked(get_settings_value(self.settings, ['integration', 'purge-queue-on-start']))
        self.init_fields(fields='listener')
        self.checkBox_use_proxy.setChecked(get_settings_value(self.settings, ['proxy-server', 'use-proxy-server']))
        self.lineEdit_host.setText(get_settings_value(self.settings, ['proxy-server', 'proxy-host-ip']))
        self.lineEdit_port.setText(str(get_settings_value(self.settings, ['proxy-server', 'proxy-host-port'])))
        current_proxy_protocol=get_settings_value(self.settings, ['proxy-server', 'proxy-type'])
        if not current_proxy_protocol:
            self.comboBox_protocol.setCurrentText('None')
        else:
            self.comboBox_protocol.setCurrentText(current_proxy_protocol)
        self.lineEdit_username.setText(get_settings_value(self.settings, ['proxy-server', 'http-proxy-auth-username']))
        self.lineEdit_password.setText(get_settings_value(self.settings, ['proxy-server', 'http-proxy-auth-password']))
        self.init_fields(fields='proxy')
        self.checkBox_focus_by_key.setChecked(get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-with-key']))
        self.lineEdit_focus_key.setText(get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-key']))
        self.init_fields(fields='key')
        self.checkBox_focus_by_title.setChecked(get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-by-window-title']))
        self.init_fields(fields='title')
        self.lineEdit_focus_win_title.setText(get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-window-title']))
        self.checkBox_dictaphone_toggle.setChecked(get_settings_value(self.settings, ['dictaphone-key-bindings', 'enable-hotkey-listener']))
        self.label_toggle_key_combination.setEnabled(self.checkBox_dictaphone_toggle.isChecked())
        self.lineEdit_toggle_key_combination.setText(get_settings_value(self.settings, ['dictaphone-key-bindings', 'toggle-key']))
        self.lineEdit_toggle_key_combination.setEnabled(self.checkBox_dictaphone_toggle.isChecked())

        logger.debug(f"3. signals connected")
        
        ### commands
        self.listWidget_command.setCurrentRow(0)
        self.set_commands_for_destination_format(self.comboBox_destination_format.currentIndex())
        self.comboBox_destination_format.currentIndexChanged.connect(lambda: self.set_commands_for_destination_format(self.comboBox_destination_format.currentIndex()))
        self.listWidget_command.currentItemChanged.connect(lambda: self.change_commands_fields())
        self.init_fields(fields='commands')

        logger.debug(f"4. COMMANDS set up")

        ### validation
        onlyInt = QIntValidator()
        self.lineEdit_port.setValidator(onlyInt)
        ### hide tabs that are not yet implemented
        self.tabWidget.removeTab(3)
        self.tabWidget.removeTab(3)
        ### replacements
        self.reload_replacements()

        logger.debug(f"5. Replacements reloaded")

    @QtCore.pyqtSlot("QWidget*", "QWidget*")
    def on_focusChanged(self, old, now):
        """
        This is to detect events that happen when some widget lose focus. We need this to trigger 
        lineEdit content validation and to store it into self.settings. The variable self.settings
        is kept in memory so without an explicit press on the button "Shrani", the changes will be 
        lost on exit.
        """
        self.label_result_status.clear()
        # proxy
        if self.lineEdit_host == old and self.lineEdit_host.text() != get_settings_value(self.settings, ['proxy-server', 'proxy-host-ip']):
            self.settings['proxy-server']['proxy-host-ip']=self.lineEdit_host.text()
            logger.debug(f"Value of the setting 'proxy-host-ip' changed to {self.lineEdit_host.text()}")
        elif self.lineEdit_port == old and int(_nvl(self.lineEdit_port.text(),'0')) != get_settings_value(self.settings, ['proxy-server', 'proxy-host-port']):
            self.settings['proxy-server']['proxy-host-port']=self.lineEdit_port.text()
            logger.debug(f"Value of the setting 'proxy-host-port' changed to {self.lineEdit_port.text()}")
        elif self.lineEdit_username == old: 
            current_proxy_username=self.lineEdit_username.text()
            if current_proxy_username == "":
                current_proxy_username = None
            if current_proxy_username != get_settings_value(self.settings, ['proxy-server', 'http-proxy-auth-username']):
                self.settings['proxy-server']['http-proxy-auth-username']=current_proxy_username
                logger.debug(f"Value of the setting 'http-proxy-auth-username' changed to {current_proxy_username}")
        elif self.lineEdit_password == old: 
            current_proxy_password=self.lineEdit_password.text()
            if current_proxy_password == "":
                current_proxy_password = None
            if current_proxy_password != get_settings_value(self.settings, ['proxy-server', 'http-proxy-auth-password']):
                self.settings['proxy-server']['http-proxy-auth-password']=current_proxy_password
                logger.debug(f"Value of the setting 'http-proxy-auth-password' changed to {current_proxy_password}")
        # autofocus
        elif self.lineEdit_focus_key == old and self.lineEdit_focus_key.text() != get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-key']):
            self.settings['general']['autofocus']['autofocus-key']=self.lineEdit_focus_key.text()
            logger.debug(f"Value of the setting 'autofocus-key' changed to {self.lineEdit_focus_key.text()}")
        elif self.lineEdit_focus_win_title == old and self.lineEdit_focus_win_title.text() != get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-window-title']):
            self.settings['general']['autofocus']['autofocus-window-title']=self.lineEdit_focus_win_title.text()
            logger.debug(f"Value of the setting 'autofocus-window-title' changed to {self.lineEdit_focus_win_title.text()}")
        # dictaphone
        elif self.lineEdit_toggle_key_combination == old and self.lineEdit_toggle_key_combination.text() != get_settings_value(self.settings, ['dictaphone-key-bindings', 'toggle-key']):
            self.settings['dictaphone-key-bindings']['toggle-key']=self.lineEdit_toggle_key_combination.text()
            logger.debug(f"Value of the setting 'toggle-key' for dictaphone changed to {self.lineEdit_toggle_key_combination.text()}")
        # commands settings
        elif self.lineEdit_keypress == old and self.listWidget_command.currentItem():
            _current_dest_idx = self.comboBox_destination_format.currentIndex()
            _current_cmd = self.__get_command_label(self.commands, self.listWidget_command.currentItem().text())
            _current_cmd_settings = get_settings_value(self.settings, ['commands', _current_cmd, self.destination_formats[_current_dest_idx]]) or dict()
            if self.lineEdit_keypress.text() != _current_cmd_settings.get('keyboard_press', False):
                self.settings['commands'][_current_cmd][self.destination_formats[_current_dest_idx]]['keyboard_press']=self.lineEdit_keypress.text()
                logger.debug(f"Value of the setting 'keyboard_press' on command '{_current_cmd}' for destination format '{self.destination_formats[_current_dest_idx]}' changed to {self.lineEdit_keypress.text()}")
        elif self.lineEdit_copy2clip == old and self.listWidget_command.currentItem():
            _current_dest_idx = self.comboBox_destination_format.currentIndex()
            _current_cmd = self.__get_command_label(self.commands, self.listWidget_command.currentItem().text())
            _current_cmd_settings = get_settings_value(self.settings, ['commands', _current_cmd, self.destination_formats[_current_dest_idx]]) or dict()
            if _nvl(self.lineEdit_copy2clip.text().replace('<nl>', '\n').replace('<tab>', '\t'), None) != _current_cmd_settings.get('copyclip', False):
                new_value = self.lineEdit_copy2clip.text().replace('<nl>', '\n').replace('<tab>', '\t')
                self.settings['commands'][_current_cmd][self.destination_formats[_current_dest_idx]]['copyclip']=new_value
                logger.debug(f"Value of the setting 'copyclip' on command '{_current_cmd}' for destination format '{self.destination_formats[_current_dest_idx]}' changed to {new_value}")
        elif self.lineEdit_copy2buf == old and self.listWidget_command.currentItem():
            _current_dest_idx = self.comboBox_destination_format.currentIndex()
            _current_cmd = self.__get_command_label(self.commands, self.listWidget_command.currentItem().text())
            _current_cmd_settings = get_settings_value(self.settings, ['commands', _current_cmd, self.destination_formats[_current_dest_idx]]) or dict()
            if _nvl(self.lineEdit_copy2buf.text().replace('<nl>', '\n').replace('<tab>', '\t'), None) != _current_cmd_settings.get('copy2buffer', False):
                new_value = self.lineEdit_copy2buf.text().replace('<nl>', '\n').replace('<tab>', '\t')
                self.settings['commands'][_current_cmd][self.destination_formats[_current_dest_idx]]['copy2buffer']=new_value
                logger.debug(f"Value of the setting 'copy2buffer' on command '{_current_cmd}' for destination format '{self.destination_formats[_current_dest_idx]}' changed to {new_value}")
        elif self.spinBox_repeat == old and self.listWidget_command.currentItem():
            _current_dest_idx = self.comboBox_destination_format.currentIndex()
            _current_cmd = self.__get_command_label(self.commands, self.listWidget_command.currentItem().text())
            _current_cmd_settings = get_settings_value(self.settings, ['commands', _current_cmd, self.destination_formats[_current_dest_idx]]) or dict()
            if _nvl(self.spinBox_repeat.value(), None) != _current_cmd_settings.get('repeat', False):
                new_value = self.spinBox_repeat.value()
                self.settings['commands'][_current_cmd][self.destination_formats[_current_dest_idx]]['repeat']=new_value
                logger.debug(f"Value of the setting 'repeat' on command '{_current_cmd}' for destination format '{self.destination_formats[_current_dest_idx]}' changed to {new_value}")

        # check if settings have changed
        if self.settings != self.settings_orig:
            self.settings_changed = True
        else:
            self.settings_changed = False 

    def init_fields(self, fields='key'):
        if fields=='key':
            self.lineEdit_focus_key.setEnabled(self.checkBox_focus_by_key.isChecked())
        elif fields=='title':
            self.lineEdit_focus_win_title.setEnabled(self.checkBox_focus_by_title.isChecked())
        elif fields=='listener':
            self.checkBox_purge_queue.setEnabled(self.checkBox_listen_external.isChecked())
        elif fields=='proxy':
            self.lineEdit_host.setEnabled(self.checkBox_use_proxy.isChecked())
            self.lineEdit_port.setEnabled(self.checkBox_use_proxy.isChecked())
            self.comboBox_protocol.setEnabled(self.checkBox_use_proxy.isChecked())
            if not self.checkBox_use_proxy.isChecked():
                self.comboBox_protocol.setStyleSheet("background-color: white; border-style: solid; border-width: 1px; border-color: lightgrey")
            else:
                self.comboBox_protocol.setStyleSheet("background-color: white; border-style: solid; border-width: 1px; border-color: grey")
            self.lineEdit_username.setEnabled(self.checkBox_use_proxy.isChecked())
            self.lineEdit_password.setEnabled(self.checkBox_use_proxy.isChecked())
            self.label_host.setEnabled(self.checkBox_use_proxy.isChecked())
            self.label_port.setEnabled(self.checkBox_use_proxy.isChecked())
            self.label_protocol.setEnabled(self.checkBox_use_proxy.isChecked())
            self.label_username.setEnabled(self.checkBox_use_proxy.isChecked())
            self.label_password.setEnabled(self.checkBox_use_proxy.isChecked())
        elif fields=='command':
            if self.listWidget_command.currentItem():
                _current_dest_idx = self.comboBox_destination_format.currentIndex()
                _current_cmd = self.__get_command_label(self.commands, self.listWidget_command.currentItem().text())
                _current_cmd_settings = get_settings_value(self.settings, ['commands', _current_cmd, self.destination_formats[_current_dest_idx]])
                if _current_cmd_settings:
                    self.lineEdit_keypress.setText(_current_cmd_settings.get('keyboard_press', ''))
                    if _current_cmd_settings.get('repeat', False):
                        self.spinBox_repeat.setValue(int(_current_cmd_settings.get('repeat')))
                    else:
                        self.spinBox_repeat.setValue(1)
                    if _current_cmd_settings.get('copyclip', ''):
                        self.lineEdit_copy2clip.setText(_current_cmd_settings.get('copyclip', '').replace('\n', '<nl>').replace('\t', '<tab>'))
                    else:
                        self.lineEdit_copy2clip.setText('')
                    if _current_cmd_settings.get('copy2buffer', ''):
                        self.lineEdit_copy2buf.setText(_current_cmd_settings.get('copy2buffer', '').replace('\n', '<nl>').replace('\t', '<tab>'))
                    else:
                        self.lineEdit_copy2buf.setText('')
                   
            if not self.listWidget_command.currentItem() or not _current_cmd_settings:
                self.lineEdit_keypress.clear()
                self.spinBox_repeat.clear()
                self.lineEdit_copy2clip.clear()
                self.lineEdit_copy2buf.clear()

    def close_window(self):
        if self.settings_changed:
            if show_message_box(
                message='Nastavitve so bile spremenjene. Če jih ne shranite, se bodo ob naslednjem zagonu aplikacije naložile stare nastavitve!',
                box_title='Neshranjene spremembe',
                button_yes_text='Shrani in zapri',
                button_no_text='Zapri',
                box_type='Question',
                num_of_buttons=2
                ) == 1:
                if save_config(self.settings):
                    self.settings_orig = copy.deepcopy(self.settings)
                else:
                    show_message_box(
                        message='Nastavitve niso bile shranjene! Obrnite se na skrbnika aplikacije!',
                        box_title='Napaka',
                        button_yes_text='Zapri',
                        box_type='Critical',
                        num_of_buttons=1
                        )

        if self.replacements_changed():
            if show_message_box(
                message='Zamenjave so bile spremenjene. Jih želite shraniti?',
                box_title='Neshranjene zamenjave',
                button_yes_text='Shrani in zapri',
                button_no_text='Zapri',
                box_type='Question',
                num_of_buttons=2,
                ) == 1:
                res = self.update_user_replacements_on_backend()
                if res['post_error'] + res['del_error'] > 0:
                    show_message_box(
                        message='Pri shranjevanju zamenjav so se zgodile napake! Nekatere zamenjave niso bilo shranjene oz. brisane! Obrnite se na skrbnika aplikacije.',
                        box_title='Napaka',
                        button_yes_text='Zapri',
                        box_type='Critical',
                        num_of_buttons=1
                        )
        
        self.gui_object.close()

    def change_commands_fields(self):
        self.display_command_help()
        self.init_fields(fields='command')

    def display_command_help(self):
        if self.listWidget_command.currentItem():
            self.textEdit_command_meaning.setPlainText(self.commands_help.get(self.listWidget_command.currentItem().text(), ""))

    def __get_command_label(self, d, val):
        res = [k for k, v in d.items() if v == val]
        if len(res)==0:
            return ""
        else:
            return res[0]
    
    def set_commands_for_destination_format(self, destination_format):
        self.listWidget_command.clear()
        self.listWidget_command.addItems([self.commands[cmd] for cmd in get_settings_value(self.settings, ['legal-commands', self.destination_formats[destination_format]])])

    def save_changes(self, fields):
        """
        Whenever a checkBox or comboBox is changed, this will fire and changes will be saved to self.settings variable
        Notice that changes to content of lineEdit components are handled with the 'on_focusChanged' handler
        """
        self.label_result_status.clear()
        # post-processing
        if fields=='numbers' and self.checkBox_numbers.isChecked() != get_settings_value(self.settings, ['general', 'do-postprocessing', 'numbers']):
            self.settings['general']['do-postprocessing']['numbers']=self.checkBox_numbers.isChecked()
            logger.debug(f"Value of the setting 'numbers' changed to {self.checkBox_numbers.isChecked()}")
        elif fields=='spp' and self.checkBox_spp.isChecked() != get_settings_value(self.settings, ['general', 'do-postprocessing', 'spp-codes']):
            self.settings['general']['do-postprocessing']['spp-codes']=self.checkBox_spp.isChecked()
            logger.debug(f"Value of the setting 'spp' changed to {self.checkBox_spp.isChecked()}")
        elif fields=='capitalization' and self.checkBox_capitalization.isChecked() != get_settings_value(self.settings, ['general', 'do-postprocessing', 'capitalize-sentences']):
            self.settings['general']['do-postprocessing']['capitalize-sentences']=self.checkBox_capitalization.isChecked()
            logger.debug(f"Value of the setting 'capitalization' changed to {self.checkBox_capitalization.isChecked()}")
        elif fields=='beep' and self.checkBox_beep_on_final.isChecked() != get_settings_value(self.settings, ['general', 'beep-on-finals']):
            self.settings['general']['beep-on-finals']=self.checkBox_beep_on_final.isChecked()
            logger.debug(f"Value of the setting 'beep-on-finals' changed to {self.checkBox_beep_on_final.isChecked()}")
        # notes
        elif fields=='notes' and self.checkBox_format_notes.isChecked() != get_settings_value(self.settings, ['general', 'do-postprocessing', 'format-notes']):
            self.settings['general']['do-postprocessing']['format-notes']=self.checkBox_format_notes.isChecked()
            logger.debug(f"Value of the setting 'format-notes' changed to {self.checkBox_format_notes.isChecked()}")
        # PID dialog box
        elif fields=='pid' and self.checkBox_show_PID.isChecked() != get_settings_value(self.settings, ['general', 'show-dialog-PID']):
            self.settings['general']['show-dialog-PID']=self.checkBox_show_PID.isChecked()
            logger.debug(f"Value of the setting 'show-dialog-PID' changed to {self.checkBox_show_PID.isChecked()}")
        # proxy
        elif fields=='use-proxy' and self.checkBox_use_proxy.isChecked() != get_settings_value(self.settings, ['proxy-server', 'use-proxy-server']):
            self.settings['proxy-server']['use-proxy-server']=self.checkBox_use_proxy.isChecked()
            logger.debug(f"Value of the setting 'use-proxy-server' changed to {self.checkBox_use_proxy.isChecked()}")
        elif fields=='proxy-protocol':
            if self.comboBox_protocol.currentText() not in ('http', 'socks4', 'socks4a', 'socks5', 'socks5h'):
                self.settings['proxy-server']['proxy-type']=None
            else:
                self.settings['proxy-server']['proxy-type']=self.comboBox_protocol.currentText()
            logger.debug(f"Value of the setting 'proxy-type' changed to {self.comboBox_protocol.currentText()}")
        # autofocus
        elif fields=="key" and self.checkBox_focus_by_key.isChecked() != get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-with-key']):
            self.settings['general']['autofocus']['autofocus-with-key']=self.checkBox_focus_by_key.isChecked()
            logger.debug(f"Value of the setting 'autofocus-with-key' changed to {self.checkBox_focus_by_key.isChecked()}")
        elif fields=="title" and self.checkBox_focus_by_title.isChecked() != get_settings_value(self.settings, ['general', 'autofocus', 'autofocus-by-window-title']):
            self.settings['general']['autofocus']['autofocus-by-window-title']=self.checkBox_focus_by_title.isChecked()
            logger.debug(f"Value of the setting 'autofocus-by-window-title' changed to {self.checkBox_focus_by_title.isChecked()}")
        # listener
        elif fields=="listener" and self.checkBox_listen_external.isChecked() != get_settings_value(self.settings, ['integration', 'listen-for-external-commands']):
            self.settings['integration']['listen-for-external-commands']=self.checkBox_listen_external.isChecked()
            logger.debug(f"Value of the setting 'listen-for-external-commands' changed to {self.checkBox_listen_external.isChecked()}")
        elif fields=="purge-queue" and self.checkBox_purge_queue.isChecked() != get_settings_value(self.settings, ['integration', 'purge-queue-on-start']):
            self.settings['integration']['purge-queue-on-start']=self.checkBox_purge_queue.isChecked()
            logger.debug(f"Value of the setting 'purge-queue-on-start' changed to {self.checkBox_purge_queue.isChecked()}")
        # minimize to systray
        elif fields=='systray' and self.checkBox_minimize.isChecked() != get_settings_value(self.settings, ['general', 'system-tray', 'minimize-after-login']):
            self.settings['general']['system-tray']['minimize-after-login']=self.checkBox_minimize.isChecked()
            logger.debug(f"Value of the setting 'minimize-after-login' changed to {self.checkBox_minimize.isChecked()}")
        # dictaphone toggle key
        elif fields=='dictaphone' and self.checkBox_dictaphone_toggle.isChecked() != get_settings_value(self.settings, ['dictaphone-key-bindings', 'enable-hotkey-listener']):
            self.settings['dictaphone-key-bindings']['enable-hotkey-listener']=self.checkBox_dictaphone_toggle.isChecked()
            logger.debug(f"Value of the setting 'enable-hotkey-listener' for dictaphone changed to {self.checkBox_dictaphone_toggle.isChecked()}")

        # check if settings have changed
        if self.settings != self.settings_orig:
            self.settings_changed = True
        else:
            self.settings_changed = False

    def save_settings(self):
        # if tab 'Zamenjave' is selected, post replacements to BE
        if self.tabWidget.currentIndex()==2:
            logger.debug(f"Pushing changes of replacements to backend...")
            res = self.update_user_replacements_on_backend()
            if res['post_error'] + res['del_error'] > 0:
                self.label_result_status.setStyleSheet(f"color: {cl_message_failed}")
                self.label_result_status.setText(f"Pri shranjevanju zamenjav so se zgodile napake! Nekatere zamenjave niso bilo shranjene oz. brisane! Obrnite se na skrbnika aplikacije.")
            else:
                self.label_result_status.setStyleSheet(f"color: {cl_message_success}")
                self.label_result_status.setText(f"Zamenjave uspešno shranjene.")
            # refresh variables and return
            self.reload_replacements()
            return
        # if tab 'Osnovno' or 'Ukazi' selected, save settings to config
        if self.tabWidget.currentIndex() in (0, 1):
            logger.debug(f"Exporting self.settings to settings.yaml...")
            res = save_config(self.settings)
            if res:
                self.label_result_status.setStyleSheet("color: green")
                self.label_result_status.setText("Nastavitve uspešno shranjene.")
                self.settings_orig = self.settings
                self.settings_changed = False
            else:
                self.label_result_status.setStyleSheet(f"color: {cl_message_failed}")
                self.label_result_status.setText("Nastavitve niso bile shranjene! Obrnite se na skrbnika aplikacije.")

    def reload_replacements(self):
        # replacements retrieved from the backend
        self.user_replacements_from_backend = self.extract_replacements(self.stt.get_user_replacements())
        # replacements currently holded in memory - all, not just filtered
        self.current_list_of_replacements = copy.deepcopy(self.user_replacements_from_backend)
        # indexes of replacements from the memory that are in current filter
        self.label_replacement_result.clear()
        self.current_indexes = [i for i in range(0, len(self.current_list_of_replacements))]
        self.listWidget_basic_repl.clear()
        self.listWidget_basic_repl.addItems(self.current_list_of_replacements)
        # clear filter
        self.lineEdit_filter.clear()
        # position cursor on the first replacement
        if len(self.current_list_of_replacements) > 0: 
            self.listWidget_basic_repl.setCurrentRow(0)
            self.lineEdit_basic_repl.setText(self.listWidget_basic_repl.currentItem().text())

    def __valid_replacement(self, action='add'):
        # ignore replacements that already are on the list
        if action=='add':
            current_list = self.current_list_of_replacements
        elif action=='update':
            current_list = [el for el in self.current_list_of_replacements if el != self.listWidget_basic_repl.currentItem().text()]
        else:
            logger.error(f"Unknown action '{action}' in __valid_replacement() function!")
            return False

        if self.lineEdit_basic_repl.text() in current_list:
            self.label_replacement_result.setText("Ta zamenjava že obstaja.")
            self.label_replacement_result.setStyleSheet(f"color: {cl_message_failed}")
            return False
        # or source already have target
        elif self.lineEdit_basic_repl.text().split(": ")[0] in [r.split(": ")[0] for r in current_list]:
            self.label_replacement_result.setText("Ista beseda/fraza ima lahko samo eno zamenjavo.")
            self.label_replacement_result.setStyleSheet(f"color: {cl_message_failed}")
            return False
        self.label_replacement_result.clear()
        return True

    def add_replacement_to_list(self):
        if self.lineEdit_basic_repl.text() != '' and self.__valid_replacement(action='add'):
            # add item to the widget
            self.listWidget_basic_repl.addItem(self.lineEdit_basic_repl.text())
            # append item to the list variable
            self.current_list_of_replacements.append(self.lineEdit_basic_repl.text())
            # reload indexes
            self.current_indexes = [i for i in range(0, len(self.current_list_of_replacements))]
            print("index after add op:", self.current_indexes)
            # reset message label
            self.label_replacement_result.clear()

    def extract_replacements(self, replacement_response):
        """
        Creates list of replacements in the format: 'source: target'
        At the same time, class property self.backend_replacements_dict if populated
        """
        self.backend_replacements_dict = dict()
        repl_list = []
        for el in replacement_response:
            target = el['target']['text']
            item = f"{' '.join(s['text'] for s in el['source'])}: {target}"
            repl_list.append(item)
            self.backend_replacements_dict[item]=el['id']
        return repl_list

    def update_replacement_on_list(self):
        if not self.listWidget_basic_repl.currentItem():
            self.label_replacement_result.setText("Nobena zamenjava ni izbrana - kliknite na zamenjavo, ki jo želite spremeniti!")
            self.label_replacement_result.setStyleSheet(f"color: {cl_message_failed}") 
        elif self.__valid_replacement(action='update'):
            self.listWidget_basic_repl.currentItem().setText(self.lineEdit_basic_repl.text())
            self.current_list_of_replacements[self.current_indexes[self.listWidget_basic_repl.currentRow()]]=self.lineEdit_basic_repl.text()
            self.label_replacement_result.clear()

    def delete_replacement_from_list(self):
        # delete item from the list variable
        if self.listWidget_basic_repl.currentRow() != -1:
            del self.current_list_of_replacements[self.listWidget_basic_repl.currentRow()]
            # remove item from the widget
            self.listWidget_basic_repl.takeItem(self.listWidget_basic_repl.currentRow())
            # reload indexes
            self.current_indexes = [i for i in range(0, len(self.current_list_of_replacements))]
            # reset message label
            self.label_replacement_result.clear()
        else:
            self.label_replacement_result.setText("Nobena zamenjava ni izbrana - kliknite na zamenjavo, ki jo želite brisati!")
            self.label_replacement_result.setStyleSheet(f"color: {cl_message_failed}")

    def populate_list_of_replacements(self):
        if self.listWidget_basic_repl.currentItem():
            self.lineEdit_basic_repl.setText(self.listWidget_basic_repl.currentItem().text())

    def filter_replacements(self, text):
        self.listWidget_basic_repl.clear()
        self.current_indexes = []
        for c, el in enumerate(self.current_list_of_replacements):
            if text in el: 
                QListWidgetItem(el, self.listWidget_basic_repl)
                self.current_indexes.append(c)

    def update_user_replacements_on_backend(self):
        """
        Compare user replacements on the backend with current replacements
        and make necessary changes at the backend. 
        """
        def reformat_replications(d):
            """
            This will put replications into format as required by the POST endpoint
            {"id":4,"source":[{"spaceBefore":true,"text":"sončen"}],"target":{"spaceBefore":true,"text":"sočen"}},
            """
            d = dict()
            for r in repl_to_add:
                source = r.split(": ")[0]
                target = r.split(": ")[1]
                d[target] = [dict(text=s, spaceBefore=True) for s in source.split(' ')]
            return d

        # use DELETE to remove replications that were deleted
        repl_to_del = [r for r in self.user_replacements_from_backend if r not in self.current_list_of_replacements]

        del_error = 0
        del_success = 0

        for r in repl_to_del:
            if r not in self.backend_replacements_dict:
                logger.error(f'Replacement {r} cannot be deleted since it was not found among items imported from the backend!')
                continue
            else: 
                if not self.stt.delete_user_replacement(self.backend_replacements_dict[r]):
                    del_error += 1
                else:
                    del_success += 1

        res = dict(del_success=del_success, del_error=del_error)

        # use POST to sync replacements that were changed or added
        repl_to_add = [r for r in self.current_list_of_replacements if r not in self.user_replacements_from_backend]
        repl_dict = reformat_replications(repl_to_add)

        post_error = 0
        post_success = 0

        for el in repl_dict.keys():
            json_data = dict(source=repl_dict[el], target=dict(text=el, spaceBefore=True))
            if not self.stt.post_user_replacement(json_data):
                post_error += 1
                print(json_data)
            else:
                post_success += 1
        
        res.update(post_success=post_success, post_error=post_error)
        
        print(res)

        return res

    def replacements_changed(self):
        repl_to_del = [r for r in self.user_replacements_from_backend if r not in self.current_list_of_replacements]
        repl_to_add = [r for r in self.current_list_of_replacements if r not in self.user_replacements_from_backend]
        if len(repl_to_add) + len(repl_to_del) > 0:
            return True
        else:
            return False

    def textEditTest(self):
        self.textEdit.setText("To je <b>tekst</b>")

    
